﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using Mono.Cecil;
using System.Reflection;

namespace NamePatcher
{

	class NamePatcher
	{
		static void DrawUsage ()
		{
			Console.WriteLine ("Usage :");
			Console.WriteLine ("NamePatcher [input file]    Patches an Assembly and creates a backup of the original file.");
		}

		static bool TryArgs (string[] args)
		{
			return args.Length == 1 && (File.Exists (args [0]));
		}

		static bool makeBackup (string input)
		{
			string backupFile_base = input + ".bck";
			string backupFile = backupFile_base;
			int backupIndex = 0;
			while (File.Exists(backupFile)) {
				backupFile = backupFile_base + "." + (++backupIndex);
				if (backupIndex > 10) {
					return false;
				}
			}
			try {
				File.Copy (input, backupFile);
			} catch (Exception e) {
				Console.WriteLine ("Unable to create backup file : ");
				Console.WriteLine (e.ToString ());
				return false;
			}
			return true;
		}

		static DefaultAssemblyResolver getAssemblyResolver (string path)
		{
			DefaultAssemblyResolver resolver = null;
			int lastSlash = path.LastIndexOf ("\\");
			if (lastSlash == -1)
				lastSlash = path.LastIndexOf ("/");
			if (lastSlash != -1) {
				string inputPath = path.Substring (0, lastSlash);
				resolver = new DefaultAssemblyResolver ();
				resolver.AddSearchDirectory (inputPath);
			}
			return resolver;
		}

		static void Main (string[] args)
		{
			Console.WriteLine ("NamePatcher for 7dtd's Assembly-CSharp.dll [by DerPopo, modified by Alloc]");
			if (!TryArgs (args)) {
				DrawUsage ();
				return;
			}
			string dllPath = args [0];

			if (!makeBackup (dllPath)) {
				Console.WriteLine ("Could not create a backup file (maybe too many old backups?)");
				return;
			}


			AssemblyDefinition input;
			try {
				input = AssemblyDefinition.ReadAssembly (dllPath, new ReaderParameters { AssemblyResolver = getAssemblyResolver(dllPath), });
			} catch (Exception e) {
				Console.WriteLine ("Unable to load the input file : ");
				Console.WriteLine (e.ToString ());
				return;
			}
			Console.WriteLine ();
			Console.WriteLine ("Patching assembly " + dllPath + " (" + input.Modules.Count + " module[s]) ...");



			ManualPatches.applyManualPatches (input.MainModule);


			try {
				foreach (ModuleDefinition mdef in input.Modules) {
					Console.WriteLine ("Patching module " + mdef.Name + " (" + mdef.Types.Count + " type[s]) ...");
					foreach (TypeDefinition tdef in mdef.Types) {
						NameNormalizer.CheckNames (tdef);
					}
				}
			} catch (Exception e) {
				Console.WriteLine ();
				Console.WriteLine ("Unable to patch the assembly : ");
				Console.WriteLine (e.ToString ());
				return;
			}
			Console.WriteLine ("Finalizing patches...");

			try {
				NameNormalizer.FinalizeNormalizing ();
			} catch (Exception e) {
				Console.WriteLine ();
				Console.WriteLine ("Unable to finalize patching : ");
				Console.WriteLine (e.ToString ());
				return;
			}
			NameNormalizer.clnamestomod.Clear ();
			NameNormalizer.vclasses.Clear ();
			Console.WriteLine ("Saving the patched assembly ...");

			try {
				input.Write (dllPath);
			} catch (Exception e) {
				Console.WriteLine ();
				Console.WriteLine ("Unable to save the assembly : ");
				Console.WriteLine (e.ToString ());
				return;
			}

			Console.WriteLine ();
			Console.WriteLine ("Success.");
		}
	}
}
