using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using UnityEngine;

namespace AllocsFixes.MapRendering
{
	public class MapRendering
	{
		private static MapRendering instance;

		public static MapRendering Instance {
			get {
				if (instance == null) {
					instance = new MapRendering ();
				}
				return instance;
			}
		}

		private RegionFileManager rfm;
		private Texture2D fullMapTexture = null;
		private MapRenderBlockBuffer[] zoomLevelBuffers = new MapRenderBlockBuffer[Constants.ZOOMLEVELS];
		private Color[] chunkColors = new Color[Constants.MAP_CHUNK_SIZE * Constants.MAP_CHUNK_SIZE];
		private System.Timers.Timer chunkSaveTimer = new System.Timers.Timer (500);
		private bool renderingFullMap = false;
		public static bool renderingEnabled = true;

		private MapRendering ()
		{
			Constants.MAP_DIRECTORY = StaticDirectories.GetSaveGameDir () + "/map";

			for (int i = 0; i < Constants.ZOOMLEVELS; i++) {
				zoomLevelBuffers [i] = new MapRenderBlockBuffer (i);
			}

			chunkSaveTimer.AutoReset = false;
			chunkSaveTimer.Elapsed += new System.Timers.ElapsedEventHandler (SaveAllBlockMaps);
		}

		public static void RenderSingleChunk (Chunk chunk)
		{
			if (renderingEnabled) {
				ThreadPool.QueueUserWorkItem ((o) =>
				{
					try {
						if (!Instance.renderingFullMap) {
							Chunk c = (Chunk)o;
							Vector3i cPos = c.GetWorldPos ();
							Vector2i cPos2 = new Vector2i (cPos.x / Constants.MAP_CHUNK_SIZE, cPos.z / Constants.MAP_CHUNK_SIZE);
							Instance.RenderChunk (c, cPos2);
							Instance.chunkSaveTimer.Stop ();
							Instance.chunkSaveTimer.Start ();
						}
					} catch (Exception e) {
						Log.Out ("Exception in MapRendering.RenderSingleChunk(): " + e);
					}
				}, chunk);
			}
		}

		public void RenderFullMap ()
		{
			MicroStopwatch microStopwatch = new MicroStopwatch ();

			string regionSaveDir = StaticDirectories.GetSaveGameRegionDir ();
			rfm = new RegionFileManager (regionSaveDir, regionSaveDir, 1, false);

			if (Directory.Exists (Constants.MAP_DIRECTORY))
				Directory.Delete (Constants.MAP_DIRECTORY, true);

			renderingFullMap = true;

			Vector2i minChunk, maxChunk;
			Vector2i minPos, maxPos;
			int widthChunks, heightChunks, widthPix, heightPix;
			getWorldExtent (out minChunk, out maxChunk, out minPos, out maxPos, out widthChunks, out heightChunks, out widthPix, out heightPix);

			Log.Out (String.Format ("RenderMap: min: {0}, max: {1}, minPos: {2}, maxPos: {3}, w/h: {4}/{5}, wP/hP: {6}/{7}",
			                        minChunk.ToString (), maxChunk.ToString (),
			                        minPos.ToString (), maxPos.ToString (),
			                        widthChunks, heightChunks,
			                        widthPix, heightPix)
			);

			if (widthPix <= 8000 && heightPix <= 8000)
				fullMapTexture = new Texture2D (widthPix, heightPix);

			Vector2i curFullMapPos;
			Vector2i curChunkPos;
			for (curFullMapPos.x = 0; curFullMapPos.x < widthPix; curFullMapPos.x += Constants.MAP_CHUNK_SIZE) {
				for (curFullMapPos.y = 0; curFullMapPos.y < heightPix; curFullMapPos.y += Constants.MAP_CHUNK_SIZE) {
					curChunkPos.x = (curFullMapPos.x / Constants.MAP_CHUNK_SIZE) + minChunk.x;
					curChunkPos.y = (curFullMapPos.y / Constants.MAP_CHUNK_SIZE) + minChunk.y;

					RenderChunk (curChunkPos, curFullMapPos);
				}
				Log.Out (String.Format ("RenderMap: {0}/{1} ({2}%)", curFullMapPos.x, widthPix, (int)((float)curFullMapPos.x / widthPix * 100)));
			}
			SaveAllBlockMaps (null, null);

			rfm = null;

			if (fullMapTexture != null) {
				byte[] array = fullMapTexture.EncodeToPNG ();
				File.WriteAllBytes (Constants.MAP_DIRECTORY + "/map.png", array);
				Texture2D.Destroy (fullMapTexture);
				fullMapTexture = null;
			}

			renderingFullMap = false;

			Log.Out ("Generating map took: " + microStopwatch.ElapsedMilliseconds + " ms");
			Log.Out ("World extent: " + minPos + " - " + maxPos);
		}

		private int saveCount = 0;
		private long renderCount = 0;

		private void SaveAllBlockMaps (object source, System.Timers.ElapsedEventArgs e)
		{
			Monitor.Enter (zoomLevelBuffers);
			try {
				Log.Out ("------- SaveAllBlockMaps " + ++saveCount + " - " + renderCount);
				for (int i = 0; i < Constants.ZOOMLEVELS; i++) {
					zoomLevelBuffers [i].SaveBlock ();
				}
			} finally {
				Monitor.Exit (zoomLevelBuffers);
			}
		}

		private bool RenderChunk (Vector2i chunkPos, Vector2i fullMapPos = default(Vector2i))
		{
			long chunkKey = WorldChunkCache.MakeChunkKey (chunkPos.x, chunkPos.y);
			if (rfm.ContainsChunkSync (chunkKey)) {
				Chunk chunk = rfm.GetChunkSync (chunkKey);
				return RenderChunk (chunk, chunkPos, fullMapPos);
			}
			return false;
		}

		private bool RenderChunk (Chunk chunk, Vector2i chunkPos, Vector2i fullMapPos = default(Vector2i))
		{
			Monitor.Enter (zoomLevelBuffers);
			try {
				renderCount++;
				ushort[] mapColors = chunk.GetMapColors ();
				if (mapColors != null) {
					Vector2i block, blockOffset;
					getBlockNumber (chunkPos, out block, out blockOffset, Constants.MAP_BLOCK_TO_CHUNK_DIV, Constants.MAP_CHUNK_SIZE);

					for (int i_colors = 0; i_colors < mapColors.Length; i_colors++) {
						chunkColors [i_colors] = shortColorToColor (mapColors [i_colors]);
					}

					zoomLevelBuffers [Constants.ZOOMLEVELS - 1].LoadBlock (block);
					zoomLevelBuffers [Constants.ZOOMLEVELS - 1].SetPart (blockOffset, Constants.MAP_CHUNK_SIZE, chunkColors);

					if (renderingFullMap && fullMapTexture != null)
						fullMapTexture.SetPixels (fullMapPos.x, fullMapPos.y, Constants.MAP_CHUNK_SIZE, Constants.MAP_CHUNK_SIZE, chunkColors);

					RenderZoomLevel (Constants.ZOOMLEVELS - 1, block);

					return true;
				}
				return false;
			} finally {
				Monitor.Exit (zoomLevelBuffers);
			}
		}

		private void RenderZoomLevel (int level, Vector2i innerBlock)
		{
			if (level > 0) {
				Vector2i block, blockOffset;
				getBlockNumber (innerBlock, out block, out blockOffset, 2, Constants.MAP_BLOCK_SIZE / 2);

				zoomLevelBuffers [level - 1].LoadBlock (block);
				zoomLevelBuffers [level - 1].SetPart (blockOffset, Constants.MAP_BLOCK_SIZE / 2, zoomLevelBuffers [level].GetHalfScaled ());

				RenderZoomLevel (level - 1, block);
			}
		}

		private void getBlockNumber (Vector2i innerPos, out Vector2i block, out Vector2i blockOffset, int scaleFactor, int offsetSize)
		{
			block.x = ((innerPos.x + 16777216) / scaleFactor) - (16777216 / scaleFactor);
			block.y = ((innerPos.y + 16777216) / scaleFactor) - (16777216 / scaleFactor);
			blockOffset.x = ((innerPos.x + 16777216) % scaleFactor) * offsetSize;
			blockOffset.y = ((innerPos.y + 16777216) % scaleFactor) * offsetSize;
		}

		private void getWorldExtent (out Vector2i minChunk, out Vector2i maxChunk,
	                                    out Vector2i minPos, out Vector2i maxPos,
	                                    out int widthChunks, out int heightChunks,
	                                    out int widthPix, out int heightPix)
		{
			long[] keys = rfm.GetAllChunkKeys ();
			int minX = Int32.MaxValue;
			int minY = Int32.MaxValue;
			int maxX = Int32.MinValue;
			int maxY = Int32.MinValue;
			foreach (long key in keys) {
				int x = WorldChunkCache.extractX (key);
				int y = WorldChunkCache.extractZ (key);

				if (x < minX)
					minX = x;
				if (x > maxX)
					maxX = x;
				if (y < minY)
					minY = y;
				if (y > maxY)
					maxY = y;
			}

			minChunk.x = minX;
			minChunk.y = minY;

			maxChunk.x = maxX;
			maxChunk.y = maxY;

			minPos.x = minX * Constants.MAP_CHUNK_SIZE;
			minPos.y = minY * Constants.MAP_CHUNK_SIZE;

			maxPos.x = maxX * Constants.MAP_CHUNK_SIZE;
			maxPos.y = maxY * Constants.MAP_CHUNK_SIZE;

			widthChunks = maxX - minX + 1;
			heightChunks = maxY - minY + 1;

			widthPix = widthChunks * Constants.MAP_CHUNK_SIZE;
			heightPix = heightChunks * Constants.MAP_CHUNK_SIZE;
		}

		private Color32 shortColorToColor32 (ushort col)
		{
			return new Color32 ((byte)((float)(col >> 10 & 31) / 31f * 255f), (byte)((float)(col >> 5 & 31) / 31f * 255f), (byte)((float)(col & 31) / 31f * 255f), 255);
		}

		private Color shortColorToColor (ushort col)
		{
			return new Color (((float)(col >> 10 & 31) / 31f), ((float)(col >> 5 & 31) / 31f), ((float)(col & 31) / 31f), 255);
		}

	}
}
