var REGIONSIZE = 512;
var CHUNKSIZE = 16;
var TILESIZE = 128;
var MAXZOOM = 4;

SDTD_Projection = {
	project: function (latlng) {
		return new L.Point(latlng.lng / 16, latlng.lat / 16);
	},

	unproject: function (point) {
		return new L.LatLng(point.y * 16, point.x * 16);
	}
};

SDTD_CRS = L.extend({}, L.CRS.Simple, {
	projection: SDTD_Projection,
	transformation: new L.Transformation(1, 0, -1, 0),

	scale: function (zoom) {
		return Math.pow(2, zoom);
	}
});

var CoordToChunk = function(latlng) {
	var x = Math.floor(((latlng.lng + 16777216) / CHUNKSIZE) - (16777216 / CHUNKSIZE));
	var y = Math.floor(((latlng.lat + 16777216) / CHUNKSIZE) - (16777216 / CHUNKSIZE));
	return L.latLng(y, x);
}

var CoordToRegion = function(latlng) {
	var x = Math.floor(((latlng.lng + 16777216) / REGIONSIZE) - (16777216 / REGIONSIZE));
	var y = Math.floor(((latlng.lat + 16777216) / REGIONSIZE) - (16777216 / REGIONSIZE));
	return L.latLng(y, x);
}

var FormatCoord = function(latlng) {
	return Math.abs(latlng.lat)+ (latlng.lat>=0 ? " N" : " S") + " / " + Math.abs(latlng.lng) + (latlng.lng>=0 ? " E" : " W");
}

var FormatRegionFileName = function(latlng) {
	return "r." + latlng.lng + "." + latlng.lat + ".7rg";
}

var map = L.map('map', {
	zoomControl: true,
	attributionControl: false,
	crs: SDTD_CRS
}).setView([0, 0], 0);

var tileLayer = L.tileLayer('../../map/{z}/{x}/{y}.png?t={time}', {
	maxZoom: MAXZOOM+1,
	minZoom: 0,
	maxNativeZoom: MAXZOOM,
	tileSize: TILESIZE,
	continuousWorld: true,
	tms: true,
	unloadInvisibleTiles: true,
	time: function() { return new Date().getTime(); }
});

var regionLayer = L.tileLayer.canvas({
	maxZoom: MAXZOOM+1,
	minZoom: 0,
	maxNativeZoom: MAXZOOM+1,
	tileSize: TILESIZE,
	continuousWorld: true
});

regionLayer.drawTile = function(canvas, tilePoint, zoom) {
	var blockWorldSize = TILESIZE * Math.pow(2, MAXZOOM-zoom);
	var tileLeft = tilePoint.x * blockWorldSize;
	var tileBottom = (-1-tilePoint.y) * blockWorldSize;
	var blockPos = L.latLng(tileBottom, tileLeft);
	
	var ctx = canvas.getContext('2d');
	
	ctx.strokeStyle = "lightblue";
	ctx.fillStyle = "lightblue";
	ctx.lineWidth = 1;
	ctx.font="14px Arial";
	
	var lineCount = blockWorldSize / REGIONSIZE;
	if (lineCount >= 1) {
		var pos = 0;
		while (pos < TILESIZE) {
			// Vertical
			ctx.beginPath();
			ctx.moveTo(pos, 0);
			ctx.lineTo(pos, TILESIZE);
			ctx.stroke();
			
			// Horizontal
			ctx.beginPath();
			ctx.moveTo(0, pos);
			ctx.lineTo(TILESIZE, pos);
			ctx.stroke();

			pos += TILESIZE / lineCount;
		}
		ctx.fillText(FormatRegionFileName(CoordToRegion(blockPos)), 4, TILESIZE-5);
	} else {
		if ((tileLeft % REGIONSIZE) == 0) {
			// Vertical
			ctx.beginPath();
			ctx.moveTo(0, 0);
			ctx.lineTo(0, TILESIZE);
			ctx.stroke();
		}
		if ((tileBottom % REGIONSIZE) == 0) {
			// Horizontal
			ctx.beginPath();
			ctx.moveTo(0, TILESIZE);
			ctx.lineTo(TILESIZE, TILESIZE);
			ctx.stroke();
		}
		if ((tileLeft % REGIONSIZE) == 0 && (tileBottom % REGIONSIZE) == 0) {
			ctx.fillText(FormatRegionFileName(CoordToRegion(blockPos)), 4, TILESIZE-5);
		}
	}

	var pos = tileLeft;

}

var baseLayers = {
    //"Map": tileLayer
};

var overlays = {
    "Region files": regionLayer
};

tileLayer.addTo(map);
L.control.layers(baseLayers, overlays, {
	collapsed: false
}).addTo(map);

map.on('mousemove', function(e) {
	L.DomUtil.get('pos').textContent = FormatCoord(e.latlng);
});
