var REGIONSIZE = 512;
var CHUNKSIZE = 16;
var TILESIZE = 128;
var MAXZOOM = 4;

var BAG_COLS = 8;
var BAG_ROWS = 4;
var BELT_COLS = 8;
var INV_ITEM_WIDTH = 58;
var INV_ITEM_HEIGHT = 40;

SDTD_Projection = {
	project: function (latlng) {
		return new L.Point(latlng.lng / 16, latlng.lat / 16);
	},

	unproject: function (point) {
		return new L.LatLng(point.y * 16, point.x * 16);
	}
};

SDTD_CRS = L.extend({}, L.CRS.Simple, {
	projection: SDTD_Projection,
	transformation: new L.Transformation(1, 0, -1, 0),

	scale: function (zoom) {
		return Math.pow(2, zoom);
	}
});

var CoordToChunk = function(latlng) {
	var x = Math.floor(((latlng.lng + 16777216) / CHUNKSIZE) - (16777216 / CHUNKSIZE));
	var y = Math.floor(((latlng.lat + 16777216) / CHUNKSIZE) - (16777216 / CHUNKSIZE));
	return L.latLng(y, x);
}

var CoordToRegion = function(latlng) {
	var x = Math.floor(((latlng.lng + 16777216) / REGIONSIZE) - (16777216 / REGIONSIZE));
	var y = Math.floor(((latlng.lat + 16777216) / REGIONSIZE) - (16777216 / REGIONSIZE));
	return L.latLng(y, x);
}

var FormatCoord = function(latlng) {
	return Math.abs(latlng.lat)+ (latlng.lat>=0 ? " N" : " S") + " / " + Math.abs(latlng.lng) + (latlng.lng>=0 ? " E" : " W");
}

var FormatRegionFileName = function(latlng) {
	return "r." + latlng.lng + "." + latlng.lat + ".7rg";
}

var map = L.map('map', {
	zoomControl: true,
	attributionControl: false,
	crs: SDTD_CRS
}).setView([0, 0], 0);

var tileLayer = L.tileLayer('../../map/{z}/{x}/{y}.png?t={time}', {
	maxZoom: MAXZOOM+1,
	minZoom: 0,
	maxNativeZoom: MAXZOOM,
	tileSize: TILESIZE,
	continuousWorld: true,
	tms: true,
	unloadInvisibleTiles: true,
	time: function() { return new Date().getTime(); }
});

var regionLayer = L.tileLayer.canvas({
	maxZoom: MAXZOOM+1,
	minZoom: 0,
	maxNativeZoom: MAXZOOM+1,
	tileSize: TILESIZE,
	continuousWorld: true
});

regionLayer.drawTile = function(canvas, tilePoint, zoom) {
	var blockWorldSize = TILESIZE * Math.pow(2, MAXZOOM-zoom);
	var tileLeft = tilePoint.x * blockWorldSize;
	var tileBottom = (-1-tilePoint.y) * blockWorldSize;
	var blockPos = L.latLng(tileBottom, tileLeft);
	
	var ctx = canvas.getContext('2d');
	
	ctx.strokeStyle = "lightblue";
	ctx.fillStyle = "lightblue";
	ctx.lineWidth = 1;
	ctx.font="14px Arial";
	
	var lineCount = blockWorldSize / REGIONSIZE;
	if (lineCount >= 1) {
		var pos = 0;
		while (pos < TILESIZE) {
			// Vertical
			ctx.beginPath();
			ctx.moveTo(pos, 0);
			ctx.lineTo(pos, TILESIZE);
			ctx.stroke();
			
			// Horizontal
			ctx.beginPath();
			ctx.moveTo(0, pos);
			ctx.lineTo(TILESIZE, pos);
			ctx.stroke();

			pos += TILESIZE / lineCount;
		}
		ctx.fillText(FormatRegionFileName(CoordToRegion(blockPos)), 4, TILESIZE-5);
	} else {
		if ((tileLeft % REGIONSIZE) == 0) {
			// Vertical
			ctx.beginPath();
			ctx.moveTo(0, 0);
			ctx.lineTo(0, TILESIZE);
			ctx.stroke();
		}
		if ((tileBottom % REGIONSIZE) == 0) {
			// Horizontal
			ctx.beginPath();
			ctx.moveTo(0, TILESIZE);
			ctx.lineTo(TILESIZE, TILESIZE);
			ctx.stroke();
		}
		if ((tileLeft % REGIONSIZE) == 0 && (tileBottom % REGIONSIZE) == 0) {
			ctx.fillText(FormatRegionFileName(CoordToRegion(blockPos)), 4, TILESIZE-5);
		}
	}

	var pos = tileLeft;

}

var playersOnlineMarkerGroup = L.layerGroup();
var playersOfflineMarkerGroup = L.layerGroup();

var baseLayers = {
    //"Map": tileLayer
};

var overlays = {
    "Region files": regionLayer,
    "Players (online) (<span id='mapControlOnlineCount'>0</span>)" : playersOnlineMarkerGroup,
    "Players (offline) (<span id='mapControlOfflineCount'>0</span>)" : playersOfflineMarkerGroup
};

tileLayer.addTo(map);
L.control.layers(baseLayers, overlays, {
	collapsed: false
}).addTo(map);

map.on('mousemove', function(e) {
	L.DomUtil.get('pos').textContent = FormatCoord(e.latlng);
});

var playersMappingList = {};

var showInv = function(steamid) {
	$.getJSON( "../api/getplayerinventory", { steamid: steamid  })
	.done(function(data) {
		$("#invPlayerName").text(playersMappingList[steamid].name);
		for (var y = 0; y < BAG_ROWS; y++) {
			for (var x = 0; x < BAG_COLS; x++) {
				if (data.bag[y*BAG_COLS+x].count > 0) {
					$("#bagField"+x+"_"+y).attr("style", "background-image: url(itemimages/" + data.bag[y*BAG_COLS+x].name + ".png);");
					$("#bagFieldText"+x+"_"+y).text(data.bag[y*BAG_COLS+x].count);
				} else {
					$("#bagField"+x+"_"+y).attr("style", "background-image: none;");
					$("#bagFieldText"+x+"_"+y).text("");
				}
			}
		}

		for (var x = 0; x < BELT_COLS; x++) {
			if (data.belt[x].count > 0) {
				$("#beltField"+x).attr("style", "background-image: url(itemimages/" + data.belt[x].name + ".png);");
				$("#beltFieldText"+x).text(data.belt[x].count);
			} else {
				$("#beltField"+x).attr("style", "background-image: none;");
				$("#beltFieldText"+x).text("");
			}
		}

		$( "#dialog-message" ).dialog({
			modal: true,
			width: BAG_COLS*INV_ITEM_WIDTH + 60,
			buttons: {
				Ok: function() {
					$( this ).dialog( "close" );
				}
			}
		});
	})
	.fail(function() {
		console.log("Error fetching player inventory");
	})
	.always(function() {
	});
};

for (var y = 0; y < BAG_ROWS; y++) {
	$("#bagTable").append("<tr id=\"bagRow"+y+"\"></tr>");
	for (var x = 0; x < BAG_COLS; x++) {
		$("#bagRow"+y).append(
			"<td class=\"invField\" id=\"bagField"+x+"_"+y+"\">" +
			"<span class=\"invFieldText\" id=\"bagFieldText"+x+"_"+y+"\"></span>" +
			"</td>");
	}
}

$("#beltTable").append("<tr id=\"beltRow0\"></tr>");
for (var x = 0; x < BELT_COLS; x++) {
	$("#beltRow0").append(
		"<td class=\"invField\" id=\"beltField"+x+"\">" +
		"<span class=\"invFieldText\" id=\"beltFieldText"+x+"\"></span>" +
		"</td>");
}

var setPlayerMarkers = function(data) {
	var online = 0;
	var offline = 0;
	$.each( data, function( key, val ) {
		var marker;
		if (playersMappingList.hasOwnProperty(val.steamid)) {
			marker = playersMappingList[val.steamid].currentPosMarker;
			marker.setLatLng([val.position.z, val.position.x]);
		} else {
			marker = L.marker([val.position.z, val.position.x]).bindPopup(
				"Player: " + val.name + "<br/>" +
				"<a onClick='showInv(\""+val.steamid+"\")'>Show inventory</a>" 
			);
			playersMappingList[val.steamid] = { online: !val.online };
		}
		if (playersMappingList[val.steamid].online != val.online) {
			if (val.online) {
				marker.setOpacity(1.0);
				playersOfflineMarkerGroup.removeLayer(marker);
				playersOnlineMarkerGroup.addLayer(marker);
			} else {
				marker.setOpacity(0.5);
				playersOnlineMarkerGroup.removeLayer(marker);
				playersOfflineMarkerGroup.addLayer(marker);
			}
		}
		val.currentPosMarker = marker;
		playersMappingList[val.steamid] = val;
		
		if (val.online)
			online++;
		else
			offline++;
	});
	$( "#mapControlOnlineCount" ).text( online );
	$( "#mapControlOfflineCount" ).text( offline );
}

var updatePlayerEvent = function() {
	$.getJSON( "../api/getplayerslocation")
	.done(setPlayerMarkers)
	.fail(function() {
		console.log("Error fetching players list");
	})
	.always(function() {
		window.setTimeout(updatePlayerEvent, 2000);
	});
}

window.setTimeout(updatePlayerEvent, 500);

