#!/bin/bash
# Version 4

# Provides common functions for 7dtd-scripts. Not intended to be run directly.

# Check if the script is run as root (exit otherwise) and load global config
checkRootLoadConf() {
	if [ `id -u` -ne 0 ]; then
		echo "This script has to be run as root!"
		exit 10
	fi
	. /etc/7dtd.conf
}

# Get the config path for the given instance
# Params:
#   1: Instance name
# Returns:
#   Config path for instance
getInstancePath() {
	echo $SDTD_BASE/instances/$1
}

# Check if the given instance name is an existing instance
# Params:
#   1: Instance name
# Returns:
#   0/1 instance not valid/valid
isValidInstance() {
	if [ -z $1 ]; then
		echo 0
	else
		if [ ! -d $(getInstancePath $1) ]; then
			echo 0
		else
			if [ ! -f $(getInstancePath $1)/config.xml ]; then
				echo 0
			else
				echo 1
			fi
		fi
	fi
}

# Check if the given instance is currently running
# Params:
#   1: Instance name
# Returns:
#   0 = not running
#   1 = running
isRunning() {
	start-stop-daemon --status --pidfile $(getInstancePath $1)/7dtd.pid
	if [ $? -eq 0 ]; then
		echo 1
	else
		echo 0
	fi
}

# Get list of defined instances
# Returns:
#   List of instances
getInstanceList() {
	for IF in $SDTD_BASE/instances/*; do
		I=`basename $IF`
		if [ $(isValidInstance $I) -eq 1 ]; then
			echo $I
		fi
	done
}

# Get the PID of the instance if it is running, 0 otherwise
# Params:
#   1: Instance name
# Returns:
#   0 if not running
#   PID otherwise
getInstancePID() {
	if [ $(isRunning $1) -eq 1 ]; then
		cat $(getInstancePath $1)/7dtd.pid
	else
		echo 0
	fi
}

# Get a single value from a serverconfig
# Params:
#   1: Instance name
#   2: Property name
# Returns:
#   Property value
getConfigValue() {
	CONF=$(getInstancePath $1)/config.xml
	$XMLSTARLET sel -t -v "/ServerSettings/property[@name='$2']/@value" $CONF
}

# Update a single value in a serverconfig
# Params:
#   1: Instance name
#   2: Property name
#   3: New value
setConfigValue() {
	CONF=$(getInstancePath $1)/config.xml
	$XMLSTARLET ed -L -u "/ServerSettings/property[@name='$2']/@value" -v "$3" $CONF
}

# Check if a given port range (baseport, baseport+1, baseport+2 each udp)
# is already in use by any instance
# Params:
#   1: Baseport
# Returns:
#   0/1 not in use/in use
checkGamePortUsed() {
	PORTMIN=$1
	PORTMAX=$(( $1 + 2 ))
	for I in $(getInstanceList); do
		CURPORTMIN=$(getConfigValue $I "ServerPort")
		CURPORTMAX=$(( $CURPORTMIN + 2 ))
		if [ $PORTMAX -ge $CURPORTMIN -a $PORTMIN -le $CURPORTMAX ]; then
			echo 1
			return
		fi
	done
	echo 0
}

# Check if a given telnet port is already in use by any instance
# Params:
#   1: Port
# Returns:
#   0/1 not in use/in use
checkTelnetPortUsed() {
	for I in $(getInstanceList); do
		CURENABLED=$(getConfigValue $I "TelnetEnabled")
		CURPORT=$(getConfigValue $I "TelnetPort")
		if [ "$CURENABLED" = "true" -a $CURPORT -eq $1 ]; then
			echo 1
			return
		fi
	done
	echo 0
}

# Send a single command to the telnet port
# Params:
#   1: Instance name
#   2: Command
# Returns:
#   String of telnet output
telnetCommand() {
	TEL_ENABLED=$(getConfigValue $1 TelnetEnabled)
	TEL_PORT=$(getConfigValue $1 TelnetPort)
	TEL_PASS=$(getConfigValue $1 TelnetPassword)	
	if [ "$TEL_ENABLED" = "true" ] && [ -n "$TEL_PASS" ]; then
		echo -e "$TEL_PASS\n$2\nexit" | nc -q 2 127.0.0.1 $TEL_PORT
	else
		echo "Telnet not enabled or no password set."
	fi
}

# Get all hook files for the given hook-name
# Params:
#   1: Hook name
# Returns:
#   Names of hook files
getHooksFor() {
	if [ -d $SDTD_BASE/hooks/$1 ]; then
		for H in $SDTD_BASE/hooks/$1/*.sh; do
			echo "$H"
		done
	fi
}

# Lowercase passed string
# Params:
#   1: String
# Returns:
#   Lowercased string
lowercase() {
	echo "${1}" | tr "[:upper:]" "[:lower:]"
}

# Prepare passed string as part of camelcase, i.e. first char upper case, others
# lowercase
# Params:
#   1: String
# Returns:
#   Transformed string
camelcasePrep() {
	echo $(echo "${1:0:1}" | tr "[:lower:]" "[:upper:]")$(echo "${1:1}" | tr "[:upper:]" "[:lower:]")
}

listCommands() {
	for C in $(declare -F | cut -d\  -f3 | grep "^sdtdCommand"\
			| grep -v "Help$"\
			| grep -v "Description$"\
			| grep -v "Expects$"); do
		CMD=$(lowercase "${C#sdtdCommand}")
		printf "%s " "$CMD"
	done
}

. /usr/local/lib/7dtd/help.sh
. /usr/local/lib/7dtd/playerlog.sh
for M in /usr/local/lib/7dtd/commands/*.sh; do
	. $M
done

