using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Steamworks;

namespace AllocsFixes
{
	public class CommonMappingFunctions
	{
		public static ConnectionManager GetConnectionManager ()
		{
			return ConnectionManager.Instance;
		}

		public static GameManager GetGameManager ()
		{
			return GetConnectionManager ().gameManager;
		}

		public static string GetPlayerName (ClientInfo _ci)
		{
			try {
				int entityId = GetConnectionManager ().mapClientToEntity [_ci.clientId];
				return GetGameManager ().World.playerEntities.dict [entityId].EntityName;
			} catch (Exception e) {
				Log.Out ("Error getting player name for ClientInfo: " + e);
			}
			return null;
		}

		public static EntityPlayer GetEntityPlayer (ClientInfo _ci)
		{
			try {
				int entityId = GetConnectionManager ().mapClientToEntity [_ci.clientId];
				return GetGameManager ().World.playerEntities.dict [entityId];
			} catch (Exception e) {
				Log.Out ("Error getting entity player for ClientInfo: " + e);
			}
			return null;
		}

		public static string GetSteamID (ClientInfo _ci)
		{
			return Steam.Authentication.Server.GetPlayerId (GetPlayerName (_ci));
		}

		public static string GetSteamID (string _playerName)
		{
			return Steam.Authentication.Server.GetPlayerId (_playerName);
		}

		public static int GetClientID (ClientInfo _ci)
		{
			if (_ci != null) {
				if (GetConnectionManager ().connectedClients.ContainsKey (_ci.clientId))
					return _ci.clientId;
			}
			return -1;
		}

		public static int GetEntityID (ClientInfo _ci)
		{
			try {
				ConnectionManager cm = GetConnectionManager ();

				if (cm.mapClientToEntity.ContainsKey (_ci.clientId))
					return cm.mapClientToEntity [_ci.clientId];
				else
					return -1;
			} catch (Exception e) {
				Log.Out ("Error getting entity ID for ClientInfo: " + e);
			}
			return -1;
		}

		public static ClientInfo GetClientInfoFromEntityID (int _entityId)
		{
			try {
				ConnectionManager cm = GetConnectionManager ();

				if (cm.mapClientToEntity.ContainsValue (_entityId)) {
					foreach (KeyValuePair<int, int> kvp in cm.mapClientToEntity) {
						if (kvp.Value == _entityId) {
							return cm.connectedClients [kvp.Key];
						}
					}
				}

				return null;
			} catch (Exception e) {
				Log.Out ("Error getting ClientInfo for entity ID: " + e);
			}
			return null;
		}

		public static ClientInfo GetClientInfoFromClientID (int _clientId)
		{
			try {
				ConnectionManager cm = GetConnectionManager ();

				if (cm.connectedClients.ContainsKey (_clientId))
					return cm.connectedClients [_clientId];
				else
					return null;
			} catch (Exception e) {
				Log.Out ("Error getting ClientInfo for client ID: " + e);
			}
			return null;
		}

		public static ClientInfo GetClientInfoFromPlayerName (string _playerName, bool ignoreColorcodes)
		{
			try {
				ConnectionManager cm = GetConnectionManager ();

				_playerName = _playerName.ToLower ();
				if (ignoreColorcodes) {
					_playerName = Regex.Replace (_playerName, "\\[[0-9a-fA-F]{6}\\]", "");
				}
				foreach (ClientInfo ci in cm.connectedClients.Values) {
					string curName = GetPlayerName (ci).ToLower ();
					if (ignoreColorcodes) {
						curName = Regex.Replace (curName, "\\[[0-9a-fA-F]{6}\\]", "");
					}
					if (curName.Equals (_playerName)) {
						return ci;
					}
				}
			} catch (Exception e) {
				Log.Out ("Error getting ClientInfo for player name: " + e);
			}
			return null;
		}

		public static ClientInfo GetClientInfoFromNameOrID (string _nameOrId, bool ignoreColorcodes)
		{
			try {
				long tempLong;
				if (_nameOrId.Length == 17 && long.TryParse (_nameOrId, out tempLong)) {
					return GetClientInfoFromSteamID (_nameOrId);
				} else {
					int entityId = -1;
					if (int.TryParse (_nameOrId, out entityId)) {
						ClientInfo ci = GetClientInfoFromEntityID (entityId);
						if (ci != null)
							return ci;
					}

					return GetClientInfoFromPlayerName (_nameOrId, ignoreColorcodes);
				}
			} catch (Exception e) {
				Log.Out ("Error getting ClientInfo for steam ID / entity ID / player name \"" + _nameOrId + "\": " + e);
			}
			return null;
		}

		public static ClientInfo GetClientInfoFromSteamID (string _steamId)
		{
			try {
				foreach (string name in Steam.Authentication.Server.usersToIDs.Keys) {
					string curId = string.Empty + Steam.Authentication.Server.usersToIDs[name].steamID.m_SteamID;
					if (curId.Equals (_steamId)) {
						return GetClientInfoFromPlayerName (name, false);
					}
				}
			} catch (Exception e) {
				Log.Out ("Error getting ClientInfo for steam ID: " + e);
			}
			return null;
		}

	}
}
