var mapinfo = {
	regionsize: 512,
	chunksize: 16,
	tilesize: 128,
	maxzoom: 4
}

function InitMap() {
	// ===============================================================================================
	// 7dtd coordinate transformations

	SDTD_Projection = {
		project: function (latlng) {
			return new L.Point(
				(latlng.lat) / Math.pow(2, mapinfo.maxzoom),
				(latlng.lng) / Math.pow(2, mapinfo.maxzoom) );
		},
		
		unproject: function (point) {
			return new L.LatLng(
				point.x * Math.pow(2, mapinfo.maxzoom),
				point.y * Math.pow(2, mapinfo.maxzoom) );
		}
	};

	SDTD_CRS = L.extend({}, L.CRS.Simple, {
		projection: SDTD_Projection,
		transformation: new L.Transformation(1, 0, -1, 0),

		scale: function (zoom) {
			return Math.pow(2, zoom);
		}
	});




	// ===============================================================================================
	// Map and basic tile layers

	var initTime = new Date().getTime();

	map = L.map('tab_map', {
		zoomControl: false, // Added by Zoomslider
		zoomsliderControl: true,
		attributionControl: false,
		crs: SDTD_CRS
	}).setView([0, 0], Math.max(0, mapinfo.maxzoom - 5));

	var tileLayer = L.tileLayer('../map/{z}/{x}/{y}.png?t={time}', {
		maxZoom: mapinfo.maxzoom + 1,
		minZoom: Math.max(0, mapinfo.maxzoom - 5),
		maxNativeZoom: mapinfo.maxzoom,
		tileSize: mapinfo.tilesize,
		continuousWorld: true,
		tms: true,
		unloadInvisibleTiles: false,
		time: initTime
	});
	
	// TileLayer w/ TMS=true fix for zoomlevel >= 8
	tileLayer._getWrapTileNum = function () {
		return L.point(0, 0);
	};

	var tileLayerMiniMap = L.tileLayer('../map/{z}/{x}/{y}.png?t={time}', {
		maxZoom: mapinfo.maxzoom,
		minZoom: 0,
		maxNativeZoom: mapinfo.maxzoom,
		tileSize: mapinfo.tilesize,
		continuousWorld: true,
		tms: true,
		unloadInvisibleTiles: false,
		time: initTime
	});







	// ===============================================================================================
	// Overlays and controls

	var playersOnlineMarkerGroup = L.markerClusterGroup({
		maxClusterRadius: function(zoom) { return zoom == mapinfo.maxzoom ? 10 : 50; }
	});
	var playersOfflineMarkerGroup = L.markerClusterGroup({
		maxClusterRadius: function(zoom) { return zoom == mapinfo.maxzoom ? 10 : 50; }
	});



	var baseLayers = {
		//"Map": tileLayer
	};

	var layerControl = L.control.layers(baseLayers, null, {
		collapsed: false
	});
	
	var layerCount = 0;


	tileLayer.addTo(map);
	new L.Control.Coordinates({}).addTo(map);
	new L.Control.ReloadTiles({layers: [tileLayer, tileLayerMiniMap]}).addTo(map);
	layerControl.addOverlay (GetRegionLayer (mapinfo), "Region files");
	var miniMap = new L.Control.MiniMap(tileLayerMiniMap, {
		zoomLevelOffset: -6,
		toggleDisplay: true
	}).addTo(map);

	if (HasPermission ("webapi.getlandclaims")) {
		layerControl.addOverlay (GetLandClaimsLayer (map, mapinfo), "Land claims");
		layerCount++;
	}
	if (HasPermission ("webapi.getplayerslocation")) {
		layerControl.addOverlay (playersOfflineMarkerGroup, "Players (offline) (<span id='mapControlOfflineCount'>0</span>)");
		layerControl.addOverlay (playersOnlineMarkerGroup, "Players (online) (<span id='mapControlOnlineCount'>0</span>)");
		layerCount++;
	}

	if (layerCount > 0) {
		layerControl.addTo(map);
	}




	var playersMappingList = {};



	// ===============================================================================================
	// Player markers

	$(".leaflet-popup-pane").on('click.action', '.inventoryButton', function(event) {
		ShowInventoryDialog ($(this).data('steamid'));
	});

	var setPlayerMarkers = function(data) {
		var online = 0;
		var offline = 0;
		$.each( data, function( key, val ) {
			var marker;
			if (playersMappingList.hasOwnProperty(val.steamid)) {
				marker = playersMappingList[val.steamid].currentPosMarker;
				marker.setLatLng([val.position.x, val.position.z]);
			} else {
				marker = L.marker([val.position.x, val.position.z]).bindPopup(
					"Player: " + val.name +
					(HasPermission ("webapi.getplayerinventory") ?
						"<br/><a class='inventoryButton' data-steamid='"+val.steamid+"'>Show inventory</a>"
						: "")
				);
				playersMappingList[val.steamid] = { online: !val.online };
			}
			if (playersMappingList[val.steamid].online != val.online) {
				if (val.online) {
					marker.setOpacity(1.0);
					playersOfflineMarkerGroup.removeLayer(marker);
					playersOnlineMarkerGroup.addLayer(marker);
				} else {
					marker.setOpacity(0.5);
					playersOnlineMarkerGroup.removeLayer(marker);
					playersOfflineMarkerGroup.addLayer(marker);
				}
			}
			val.currentPosMarker = marker;
			playersMappingList[val.steamid] = val;
		
			if (val.online)
				online++;
			else
				offline++;
		});
		$( "#mapControlOnlineCount" ).text( online );
		$( "#mapControlOfflineCount" ).text( offline );
	}

	var updatePlayerEvent = function() {
		$.getJSON( "../api/getplayerslocation")
		.done(setPlayerMarkers)
		.fail(function(jqxhr, textStatus, error) {
			console.log("Error fetching players list");
		})
		.always(function() {
			window.setTimeout(updatePlayerEvent, 2000);
		});
	}

	if (HasPermission ("webapi.getplayerslocation")) {
		window.setTimeout(updatePlayerEvent, 0);
	}


}





function StartMapModule () {
	$.getJSON( "../map/mapinfo.json")
	.done(function(data) {
		mapinfo.tilesize = data.blockSize;
		mapinfo.maxzoom = data.maxZoom;
	})
	.fail(function(jqxhr, textStatus, error) {
		console.log ("Error fetching map information");
	})
	.always(function() {
		InitMap ();
	});
}


