using AllocsFixes.PersistentData;
using System;
using System.Collections.Generic;

namespace AllocsFixes.CustomCommands
{
	public class ShowInventory : ConsoleCmdAbstract {
		public override string GetDescription () {
			return "list inventory of a given player";
		}

		public override string GetHelp () {
			return "Usage:\n" +
				"   showinventory <steam id / player name / entity id>\n" +
				"Show the inventory of the player given by his SteamID, player name or\n" +
				"entity id (as given by e.g. \"lpi\")." +
				"Note: This only shows the player's inventory after it was first sent to\n" +
				"the server which happens at least every 30 seconds.";
		}

		public override string[] GetCommands () {
			return new string[] { "showinventory", "si" };
		}

		public override void Execute (List<string> _params, CommandSenderInfo _senderInfo) {
			try {
				if (_params.Count < 1) {
					SdtdConsole.Instance.Output ("Usage: showinventory <steamid|playername|entityid>");
					return;
				}

				string steamid = PersistentContainer.Instance.Players.GetSteamID (_params [0], true);
				if (steamid == null) {
					SdtdConsole.Instance.Output ("Playername or entity/steamid id not found or no inventory saved (first saved after a player has been online for 30s).");
					return;
				}

				Player p = PersistentContainer.Instance.Players [steamid, false];
				PersistentData.Inventory inv = p.Inventory;

				SdtdConsole.Instance.Output ("Belt of player " + p.Name + ":");
				PrintInv (inv.belt);
				SdtdConsole.Instance.Output (string.Empty);

				SdtdConsole.Instance.Output ("Bagpack of player " + p.Name + ":");
				PrintInv (inv.bag);
				SdtdConsole.Instance.Output (string.Empty);

				SdtdConsole.Instance.Output ("Equipment of player " + p.Name + ":");
				PrintEquipment (inv.equipment);

			} catch (Exception e) {
				Log.Out ("Error in ShowInventory.Run: " + e);
			}
		}

		private void PrintInv (List<InvItem> _inv) {
			for (int i = 0; i < _inv.Count; i++) {
				if (_inv [i] != null) {
					if (_inv [i].quality < 0) {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0}: {1:000} * {2}", i, _inv [i].count, _inv [i].itemName));
					} else {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0}: {1:000} * {2} - quality: {3}", i, _inv [i].count, _inv [i].itemName, _inv [i].quality));
					}
					DoParts (_inv [i].parts, 1);
				}
			}
		}

		private void PrintEquipment (InvItem[] _equipment) {
			AddEquipment ("head", _equipment, XMLData.Item.EnumEquipmentSlot.Head, NGuiInvGridEquipment.EnumClothingLayer.Middle);
			AddEquipment ("eyes", _equipment, XMLData.Item.EnumEquipmentSlot.Eyes, NGuiInvGridEquipment.EnumClothingLayer.Middle);
			AddEquipment ("face", _equipment, XMLData.Item.EnumEquipmentSlot.Face, NGuiInvGridEquipment.EnumClothingLayer.Middle);

			AddEquipment ("armor", _equipment, XMLData.Item.EnumEquipmentSlot.Chest, NGuiInvGridEquipment.EnumClothingLayer.Outer);
			AddEquipment ("jacket", _equipment, XMLData.Item.EnumEquipmentSlot.Chest, NGuiInvGridEquipment.EnumClothingLayer.Middle);
			AddEquipment ("shirt", _equipment, XMLData.Item.EnumEquipmentSlot.Chest, NGuiInvGridEquipment.EnumClothingLayer.Inner);

			AddEquipment ("legarmor", _equipment, XMLData.Item.EnumEquipmentSlot.Legs, NGuiInvGridEquipment.EnumClothingLayer.Outer);
			AddEquipment ("pants", _equipment, XMLData.Item.EnumEquipmentSlot.Legs, NGuiInvGridEquipment.EnumClothingLayer.Inner);
			AddEquipment ("boots", _equipment, XMLData.Item.EnumEquipmentSlot.Feet, NGuiInvGridEquipment.EnumClothingLayer.Inner);

			AddEquipment ("gloves", _equipment, XMLData.Item.EnumEquipmentSlot.Hands, NGuiInvGridEquipment.EnumClothingLayer.Inner);
			AddEquipment ("backpack", _equipment, XMLData.Item.EnumEquipmentSlot.Back, NGuiInvGridEquipment.EnumClothingLayer.Outer);
		}

		private void AddEquipment (string _slotname, InvItem[] _items, XMLData.Item.EnumEquipmentSlot _slot, NGuiInvGridEquipment.EnumClothingLayer _layer) {
			int index = (int)_slot + (int)_layer * (int)XMLData.Item.EnumEquipmentSlot.Count;
			if (_items != null && _items [index] != null) {
				if (_items [index].quality < 0) {
					SdtdConsole.Instance.Output (string.Format ("    Slot {0:8}: {1:000}", _slotname, _items [index].itemName));
				} else {
					SdtdConsole.Instance.Output (string.Format ("    Slot {0:8}: {1:000} - quality: {2}", _slotname, _items [index].itemName, _items [index].quality));
				}
				DoParts (_items [index].parts, 1);
			}
		}

		private void DoParts (InvItem[] _parts, int _indent) {
			if (_parts != null && _parts.Length > 0) {
				string indenter = new string (' ', _indent * 4);
				for (int i = 0; i < _parts.Length; i++) {
					if (_parts [i] != null) {
						if (_parts [i].quality < 0) {
							SdtdConsole.Instance.Output (string.Format ("{0}         - {1}", indenter, _parts [i].itemName));
						} else {
							SdtdConsole.Instance.Output (string.Format ("{0}         - {1} - quality: {2}", indenter, _parts [i].itemName, _parts [i].quality));
						}
						DoParts (_parts [i].parts, _indent + 1);
					}
				}
			}
		}

	}
}
