using System;
using System.Collections.Generic;
using System.Net;
using System.Threading;

using UnityEngine;

namespace AllocsFixes.NetConnections.Servers.Web.Handlers
{
	public class ItemIconHandler : PathHandler {
		private string staticPart;
		private bool logMissingFiles;
		private Dictionary<string, byte[]> icons = new Dictionary<string, byte[]> ();
		private bool loaded = false;

		public ItemIconHandler (string staticPart, bool logMissingFiles, string moduleName = null) : base(moduleName) {
			this.staticPart = staticPart;
			this.logMissingFiles = logMissingFiles;
		}

		public override void HandleRequest (HttpListenerRequest req, HttpListenerResponse resp, WebConnection user, int permissionLevel) {
			if (!loaded) {
				if (!LoadIcons ()) {
					resp.StatusCode = (int)HttpStatusCode.NotFound;
					Log.Out ("Web:IconHandler: Could not load icons");
					return;
				}
			}

			string requestFileName = req.Url.AbsolutePath.Remove (0, staticPart.Length);
			requestFileName = requestFileName.Remove (requestFileName.LastIndexOf ('.'));

			if (icons.ContainsKey (requestFileName) && req.Url.AbsolutePath.ToLower ().EndsWith (".png")) {
				resp.ContentType = MimeType.GetMimeType (".png");

				byte[] itemIconData = icons [requestFileName];

				resp.ContentLength64 = itemIconData.Length;
				resp.OutputStream.Write (itemIconData, 0, itemIconData.Length);
			} else {
				resp.StatusCode = (int)HttpStatusCode.NotFound;
				if (logMissingFiles) {
					Log.Out ("Web:IconHandler:FileNotFound: \"" + req.Url.AbsolutePath + "\" ");
				}
				return;
			}
		}

		private bool LoadIcons () {
			lock (icons) {
				if (loaded) {
					return true;
				}

				MicroStopwatch microStopwatch = new MicroStopwatch ();

				GameObject atlasObj = GameObject.Find ("/NGUI Root (2D)/ItemIconAtlas");
				if (atlasObj == null) {
					Log.Error ("Web:IconHandler: Atlas object not found");
					loaded = true;
					return false;
				}
				DynamicUIAtlas atlas = atlasObj.GetComponent<DynamicUIAtlas> ();
				if (atlas == null) {
					Log.Error ("Web:IconHandler: Atlas component not found");
					loaded = true;
					return false;
				}

				Dictionary<string, List<Color>> tintedIcons = new Dictionary<string, List<Color>> ();
				foreach (ItemClass ic in ItemClass.list) {
					if (ic != null) {
						Color tintColor = ic.GetIconTint ();
						if (tintColor != Color.white) {
							string name = ic.GetIconName ();
							if (!tintedIcons.ContainsKey (name)) {
								tintedIcons.Add (name, new List<Color> ());
							}
							List<Color> list = tintedIcons [name];
							list.Add (tintColor);
						}
					}
				}

				Texture2D atlasTex = atlas.texture as Texture2D;

				foreach (UISpriteData data in atlas.spriteList) {
					string name = data.name;
					Texture2D tex = new Texture2D (data.width, data.height, TextureFormat.ARGB32, false);
					tex.SetPixels (atlasTex.GetPixels (data.x, atlasTex.height - data.height - data.y, data.width, data.height));

					icons.Add (name + "__FFFFFF", tex.EncodeToPNG ());

					if (tintedIcons.ContainsKey (name)) {
						foreach (Color c in tintedIcons [name]) {
							Texture2D tintedTex = new Texture2D (data.width, data.height, TextureFormat.ARGB32, false);

							for (int x = 0; x < data.width; x++) {
								for (int y = 0; y < data.height; y++) {
									tintedTex.SetPixel (x, y, tex.GetPixel (x, y) * c);
								}
							}

							icons.Add (name + "__" + AllocsUtils.ColorToHex (c), tintedTex.EncodeToPNG ());

							UnityEngine.Object.Destroy (tintedTex);
						}
					}

					UnityEngine.Object.Destroy (tex);
				}

				loaded = true;
				Log.Out ("Web:IconHandler: Icons loaded - {0} ms", microStopwatch.ElapsedMilliseconds);

				return true;
			}
		}
	}
}

