using AllocsFixes.PersistentData;
using System;
using System.Collections.Generic;

namespace AllocsFixes.CustomCommands
{
	public class ShowInventory : ConsoleCmdAbstract {

		public override string GetDescription () {
			return "list inventory of a given player";
		}

		public override string GetHelp () {
			return "Usage:\n" +
				"   showinventory <steam id / player name / entity id>\n" +
				"Show the inventory of the player given by his SteamID, player name or\n" +
				"entity id (as given by e.g. \"lpi\")." +
				"Note: This only shows the player's inventory after it was first sent to\n" +
				"the server which happens at least every 30 seconds.";
		}

		public override string[] GetCommands () {
			return new string[] { "showinventory", "si" };
		}

		public override void Execute (List<string> _params, CommandSenderInfo _senderInfo) {
			try {
				if (_params.Count < 1) {
					SdtdConsole.Instance.Output ("Usage: showinventory <steamid|playername|entityid>");
					return;
				}

				string steamid = PersistentContainer.Instance.Players.GetSteamID (_params [0], true);
				if (steamid == null) {
					SdtdConsole.Instance.Output ("Playername or entity/steamid id not found or no inventory saved (first saved after a player has been online for 30s).");
					return;
				}

				Player p = PersistentContainer.Instance.Players [steamid, false];
				PersistentData.Inventory inv = p.Inventory;

				SdtdConsole.Instance.Output ("Belt of player " + p.Name + ":");
				PrintInv (inv.belt);
				SdtdConsole.Instance.Output (string.Empty);

				SdtdConsole.Instance.Output ("Bagpack of player " + p.Name + ":");
				PrintInv (inv.bag);
				SdtdConsole.Instance.Output (string.Empty);

				SdtdConsole.Instance.Output ("Equipment of player " + p.Name + ":");
				PrintEquipment (inv.equipment);

			} catch (Exception e) {
				Log.Out ("Error in ShowInventory.Run: " + e);
			}
		}

		private void PrintInv (List<InvItem> _inv) {
			for (int i = 0; i < _inv.Count; i++) {
				if (_inv [i] != null) {
					if (_inv [i].quality < 0) {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0}: {1:000} * {2}", i, _inv [i].count, _inv [i].itemName));
					} else {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0}: {1:000} * {2} - quality: {3}", i, _inv [i].count, _inv [i].itemName, _inv [i].quality));
					}
					DoParts (_inv [i].parts, 1);
				}
			}
		}

		private void PrintEquipment (InvItem[] _equipment) {
			AddEquipment ("head", _equipment, EquipmentSlots.Headgear);
			AddEquipment ("eyes", _equipment, EquipmentSlots.Eyewear);
			AddEquipment ("face", _equipment, EquipmentSlots.Face);

			AddEquipment ("armor", _equipment, EquipmentSlots.ChestArmor);
			AddEquipment ("jacket", _equipment, EquipmentSlots.Jacket);
			AddEquipment ("shirt", _equipment, EquipmentSlots.Shirt);

			AddEquipment ("legarmor", _equipment, EquipmentSlots.LegArmor);
			AddEquipment ("pants", _equipment, EquipmentSlots.Legs);
			AddEquipment ("boots", _equipment, EquipmentSlots.Feet);

			AddEquipment ("gloves", _equipment, EquipmentSlots.Hands);
		}

		private void AddEquipment (string _slotname, InvItem[] _items, EquipmentSlots _slot) {
			int[] slotindices = XUiM_PlayerEquipment.GetSlotIndicesByEquipmentSlot (_slot);

			for (int i = 0; i < slotindices.Length; i++) {
				if (_items != null && _items [slotindices [i]] != null) {
					InvItem item = _items [slotindices [i]];
					if (item.quality < 0) {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0:8}: {1:000}", _slotname, item.itemName));
					} else {
						SdtdConsole.Instance.Output (string.Format ("    Slot {0:8}: {1:000} - quality: {2}", _slotname, item.itemName, item.quality));
					}
					DoParts (_items [slotindices [i]].parts, 1);
					return;
				}
			}
		}

		private void DoParts (InvItem[] _parts, int _indent) {
			if (_parts != null && _parts.Length > 0) {
				string indenter = new string (' ', _indent * 4);
				for (int i = 0; i < _parts.Length; i++) {
					if (_parts [i] != null) {
						if (_parts [i].quality < 0) {
							SdtdConsole.Instance.Output (string.Format ("{0}         - {1}", indenter, _parts [i].itemName));
						} else {
							SdtdConsole.Instance.Output (string.Format ("{0}         - {1} - quality: {2}", indenter, _parts [i].itemName, _parts [i].quality));
						}
						DoParts (_parts [i].parts, _indent + 1);
					}
				}
			}
		}

	}
}
