using System;
using System.Collections.Generic;
using System.Reflection;

namespace AllocsFixes.CustomCommands
{
	public class RepairChunkDensity : ConsoleCmdAbstract {
		public override string GetDescription () {
			return "check and optionally fix densities of a chunk";
		}

		public override string GetHelp () {
			return "This command is used to check if the densities of blocks in a chunk match the actual block type.\n" +
				"If there is a mismatch it can lead to the chunk rendering incorrectly or not at all, typically\n" +
				"indicated by the error message \"Failed setting triangles. Some indices are referencing out of\n" +
				"bounds vertices.\". It can also fix such mismatches within a chunk.\n" +
				"Usage:\n" +
				"  1. repairchunkdensity <x> <z>\n" +
				"  2. repairchunkdensity <x> <z> fix\n" +
				"1. Just checks the chunk and prints mismatched to the server log. x and z are the coordinates of any\n" +
				"   block within the chunk to check.\n" +
				"2. Repairs any mismatch found in the chunk.";
		}

		public override string[] GetCommands () {
			return new string[] { "repairchunkdensity", "rcd" };
		}

		public override void Execute (List<string> _params, CommandSenderInfo _senderInfo) {
			try {
				if (_params.Count < 2 || _params.Count > 3) {
					SdtdConsole.Instance.Output ("Wrong number of arguments, expected 2 or 3, found " + _params.Count + ".");
					return;
				} else {
					int x = int.MinValue;
					int z = int.MinValue;

					int.TryParse (_params [0], out x);
					int.TryParse (_params [1], out z);

					if (x == int.MinValue || z == int.MinValue) {
						SdtdConsole.Instance.Output ("At least one of the given coordinates is not a valid integer");
						return;
					}

					Chunk c = GameManager.Instance.World.GetChunkFromWorldPos (x, 0, z) as Chunk;
					if (c == null) {
						SdtdConsole.Instance.Output ("No chunk could be loaded from the given coordinates");
						return;
					}

					if (_params.Count == 3) {
						if (_params [2].ToLower () != "fix") {
							SdtdConsole.Instance.Output ("Three parameters given but third parameter is not \"fix\"");
							return;
						}
						c.RepairDensities ();
						c.isModified = true;
						SdtdConsole.Instance.Output ("Chunk repaired");
					} else {
						c.CheckDensities (true);
					}
				}
			} catch (Exception e) {
				Log.Out ("Error in RepairChunkDensity.Execute: " + e);
			}
		}

	}
}
