using System.Collections.Generic;
using AllocsFixes.NetConnections.Servers.Web;

namespace AllocsFixes.CustomCommands {
	public class WebPermissionsCmd : ConsoleCmdAbstract {
		public override string[] GetCommands () {
			return new[] {"webpermission"};
		}

		public override string GetDescription () {
			return "Manage web permission levels";
		}

		public override string GetHelp () {
			return "Set/get permission levels required to access a given web functionality. Default\n" +
			       "level required for functions that are not explicitly specified is 0.\n" +
			       "Usage:\n" +
			       "   webpermission add <webfunction> <level>\n" +
			       "   webpermission remove <webfunction>\n" +
			       "   webpermission list";
		}

		public override void Execute (List<string> _params, CommandSenderInfo _senderInfo) {
			if (_params.Count >= 1) {
				switch (_params [0].ToLower ()) {
					case "add":
						ExecuteAdd (_params);
						break;
					case "remove":
						ExecuteRemove (_params);
						break;
					case "list":
						ExecuteList ();
						break;
					default:
						SdtdConsole.Instance.Output ("Invalid sub command \"" + _params [0] + "\".");
						return;
				}
			} else {
				SdtdConsole.Instance.Output ("No sub command given.");
			}
		}

		private void ExecuteAdd (List<string> _params) {
			if (_params.Count != 3) {
				SdtdConsole.Instance.Output ("Wrong number of arguments, expected 3, found " + _params.Count + ".");
				return;
			}

			if (!WebPermissions.Instance.IsKnownModule (_params [1])) {
				SdtdConsole.Instance.Output ("\"" + _params [1] + "\" is not a valid web function.");
				return;
			}

			int level;
			if (!int.TryParse (_params [2], out level)) {
				SdtdConsole.Instance.Output ("\"" + _params [2] + "\" is not a valid integer.");
				return;
			}

			WebPermissions.Instance.AddModulePermission (_params [1], level);
			SdtdConsole.Instance.Output (string.Format ("{0} added with permission level of {1}.", _params [1], level));
		}

		private void ExecuteRemove (List<string> _params) {
			if (_params.Count != 2) {
				SdtdConsole.Instance.Output ("Wrong number of arguments, expected 2, found " + _params.Count + ".");
				return;
			}

			if (!WebPermissions.Instance.IsKnownModule (_params [1])) {
				SdtdConsole.Instance.Output ("\"" + _params [1] + "\" is not a valid web function.");
				return;
			}

			WebPermissions.Instance.RemoveModulePermission (_params [1]);
			SdtdConsole.Instance.Output (string.Format ("{0} removed from permissions list.", _params [1]));
		}

		private void ExecuteList () {
			SdtdConsole.Instance.Output ("Defined web function permissions:");
			SdtdConsole.Instance.Output ("  Level: Web function");
			foreach (WebPermissions.WebModulePermission wmp in WebPermissions.Instance.GetModules ()) {
				SdtdConsole.Instance.Output (string.Format ("  {0,5}: {1}", wmp.permissionLevel, wmp.module));
			}
		}
	}
}