using System;
using System.IO;
using AllocsFixes.FileCache;
using UnityEngine;

namespace AllocsFixes.MapRendering {
	public class MapRenderBlockBuffer {
		private readonly Texture2D blockMap = new Texture2D (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE);
		private readonly MapTileCache cache;
		private readonly Color nullColor = new Color (0, 0, 0, 0);
		private readonly Texture2D zoomBuffer = new Texture2D (1, 1);
		private readonly int zoomLevel;
		private string currentBlockMap = string.Empty;

		public MapRenderBlockBuffer (int level, MapTileCache cache) {
			zoomLevel = level;
			this.cache = cache;
		}

		public void ResetBlock () {
			currentBlockMap = string.Empty;
		}

		public void SaveBlock () {
			try {
				if (currentBlockMap.Length > 0) {
					saveTextureToFile (currentBlockMap);
				}
			} catch (Exception e) {
				Log.Out ("Exception in MapRenderBlockBuffer.SaveBlock(): " + e);
			}
		}

		public bool LoadBlock (Vector2i block) {
			bool res = false;
			lock (blockMap) {
				string folder = Constants.MAP_DIRECTORY + "/" + zoomLevel + "/" + block.x;
				string fileName = folder + "/" + block.y + ".png";
				Directory.CreateDirectory (folder);
				if (!fileName.Equals (currentBlockMap)) {
					res = true;
					SaveBlock ();
					loadTextureFromFile (fileName);
				}

				currentBlockMap = fileName;
			}

			return res;
		}

		public void SetPart (Vector2i offset, int partSize, Color32[] pixels) {
			if (offset.x + partSize > blockMap.width || offset.y + partSize > blockMap.height) {
				Log.Error (string.Format ("MapBlockBuffer[{0}].SetPart ({1}, {2}, {3}) has blockMap.size ({4}/{5})",
					zoomLevel, offset, partSize, pixels.Length, blockMap.width, blockMap.height));
				return;
			}

			blockMap.SetPixels32 (offset.x, offset.y, partSize, partSize, pixels);
		}

		public Color32[] GetHalfScaled () {
			zoomBuffer.Resize (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE);
			zoomBuffer.SetPixels32 (blockMap.GetPixels32 ());

			TextureScale.Point (zoomBuffer, Constants.MAP_BLOCK_SIZE / 2, Constants.MAP_BLOCK_SIZE / 2);

			return zoomBuffer.GetPixels32 ();
		}

		private void loadTextureFromFile (string _fileName) {
			byte[] array = cache.LoadTile (zoomLevel, _fileName);
			if (array != null && blockMap.LoadImage (array) && blockMap.height == Constants.MAP_BLOCK_SIZE &&
			    blockMap.width == Constants.MAP_BLOCK_SIZE) {
				return;
			}

			if (array != null) {
				Log.Error ("Map image tile " + _fileName + " has been corrupted, recreating tile");
			}

			if (blockMap.height != Constants.MAP_BLOCK_SIZE || blockMap.width != Constants.MAP_BLOCK_SIZE) {
				blockMap.Resize (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE);
			}

			for (int x = 0; x < Constants.MAP_BLOCK_SIZE; x++) {
				for (int y = 0; y < Constants.MAP_BLOCK_SIZE; y++) {
					blockMap.SetPixel (x, y, nullColor);
				}
			}
		}

		private void saveTextureToFile (string _fileName) {
			byte[] array = blockMap.EncodeToPNG ();
			cache.SaveTile (zoomLevel, array);
		}
	}
}