using System;
using System.IO;
using AllocsFixes.FileCache;
using Unity.Collections;
using UnityEngine;
using UnityEngine.Profiling;

namespace AllocsFixes.MapRendering {
	public class MapRenderBlockBuffer {
		private readonly Texture2D blockMap = new Texture2D (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE, TextureFormat.ARGB32, false);
		private readonly MapTileCache cache;
		private readonly Color nullColor = new Color (0, 0, 0, 0);
		private readonly Texture2D zoomBuffer = new Texture2D (1, 1, TextureFormat.ARGB32, false);
		private readonly int zoomLevel;
		private readonly string folderBase;
		
		private Vector2i currentBlockMapPos = new Vector2i (Int32.MinValue, Int32.MinValue);
		private string currentBlockMapFolder = string.Empty;
		private string currentBlockMapFilename = string.Empty;

		public MapRenderBlockBuffer (int level, MapTileCache cache) {
			zoomLevel = level;
			this.cache = cache;
			folderBase = Constants.MAP_DIRECTORY + "/" + zoomLevel + "/";
		}

		public TextureFormat FormatSelf {
			get { return blockMap.format; }
		}

		public TextureFormat FormatScaled {
			get { return zoomBuffer.format; }
		}

		public void ResetBlock () {
			currentBlockMapFolder = string.Empty;
			currentBlockMapFilename = string.Empty;
			currentBlockMapPos = new Vector2i (Int32.MinValue, Int32.MinValue);
		}

		public void SaveBlock () {
			try {
				if (currentBlockMapFilename.Length > 0) {
					saveTextureToFile (currentBlockMapFilename);
				}
			} catch (Exception e) {
				Log.Warning ("Exception in MapRenderBlockBuffer.SaveBlock(): " + e);
			}
		}

		public bool LoadBlock (Vector2i block) {
			bool res = false;
			lock (blockMap) {
				if (currentBlockMapPos != block) {
					Profiler.BeginSample ("LoadBlock.Strings");
					string folder;
					if (currentBlockMapPos.x != block.x) {
						folder = folderBase + block.x;

						Profiler.BeginSample ("LoadBlock.Directory");
						Directory.CreateDirectory (folder);
						Profiler.EndSample ();
					} else {
						folder = currentBlockMapFolder;
					}

					string fileName = folder + "/" + block.y + ".png";
					Profiler.EndSample ();
					
					if (!fileName.Equals (currentBlockMapFilename)) {
						res = true;
						SaveBlock ();
						loadTextureFromFile (fileName);
					}

					currentBlockMapFolder = folder;
					currentBlockMapPos = block;
					currentBlockMapFilename = fileName;
				}
			}

			return res;
		}

		public void SetPart (Vector2i offset, int partSize, Color32[] pixels) {
			if (offset.x + partSize > Constants.MAP_BLOCK_SIZE || offset.y + partSize > Constants.MAP_BLOCK_SIZE) {
				Log.Error (string.Format ("MapBlockBuffer[{0}].SetPart ({1}, {2}, {3}) has blockMap.size ({4}/{5})",
					zoomLevel, offset, partSize, pixels.Length, Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE));
				return;
			}

			Profiler.BeginSample ("SetPart");
			blockMap.SetPixels32 (offset.x, offset.y, partSize, partSize, pixels);
			Profiler.EndSample ();
		}

		public Color32[] GetHalfScaled () {
			Profiler.BeginSample ("HalfScaled.ResizeBuffer");
			zoomBuffer.Resize (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE);
			Profiler.EndSample ();

			Profiler.BeginSample ("HalfScaled.CopyPixels");
			if (blockMap.format == zoomBuffer.format) {
				Profiler.BeginSample ("Native");
				NativeArray<byte> dataSrc = blockMap.GetRawTextureData<byte> ();
				NativeArray<byte> dataZoom = zoomBuffer.GetRawTextureData<byte> ();
				dataSrc.CopyTo (dataZoom);
				Profiler.EndSample ();
			} else {
				Profiler.BeginSample ("GetSetPixels");
				zoomBuffer.SetPixels32 (blockMap.GetPixels32 ());
				Profiler.EndSample ();
			}
			Profiler.EndSample ();

			Profiler.BeginSample ("HalfScaled.Scale");
			TextureScale.Point (zoomBuffer, Constants.MAP_BLOCK_SIZE / 2, Constants.MAP_BLOCK_SIZE / 2);
			Profiler.EndSample ();

			Profiler.BeginSample ("HalfScaled.Return");
			Color32[] result = zoomBuffer.GetPixels32 ();
			Profiler.EndSample ();

			return result;
		}

		public void SetPartNative (Vector2i offset, int partSize, NativeArray<int> pixels) {
			if (offset.x + partSize > Constants.MAP_BLOCK_SIZE || offset.y + partSize > Constants.MAP_BLOCK_SIZE) {
				Log.Error (string.Format ("MapBlockBuffer[{0}].SetPart ({1}, {2}, {3}) has blockMap.size ({4}/{5})",
					zoomLevel, offset, partSize, pixels.Length, Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE));
				return;
			}

			Profiler.BeginSample ("SetPartNative");
			NativeArray<int> destData = blockMap.GetRawTextureData<int> ();
			
			for (int y = 0; y < partSize; y++) {
				int srcLineStartIdx = partSize * y;
				int destLineStartIdx = blockMap.width * (offset.y + y) + offset.x;
				for (int x = 0; x < partSize; x++) {
					destData [destLineStartIdx + x] = pixels [srcLineStartIdx + x];
				}
			}
			Profiler.EndSample ();
		}

		public NativeArray<int> GetHalfScaledNative () {
			Profiler.BeginSample ("HalfScaledNative.ResizeBuffer");
			zoomBuffer.Resize (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE);
			Profiler.EndSample ();

			Profiler.BeginSample ("HalfScaledNative.CopyPixels");
			if (blockMap.format == zoomBuffer.format) {
				Profiler.BeginSample ("Native");
				NativeArray<byte> dataSrc = blockMap.GetRawTextureData<byte> ();
				NativeArray<byte> dataZoom = zoomBuffer.GetRawTextureData<byte> ();
				dataSrc.CopyTo (dataZoom);
				Profiler.EndSample ();
			} else {
				Profiler.BeginSample ("GetSetPixels");
				zoomBuffer.SetPixels32 (blockMap.GetPixels32 ());
				Profiler.EndSample ();
			}
			Profiler.EndSample ();

			Profiler.BeginSample ("HalfScaledNative.Scale");
			TextureScale.Point (zoomBuffer, Constants.MAP_BLOCK_SIZE / 2, Constants.MAP_BLOCK_SIZE / 2);
			Profiler.EndSample ();

			return zoomBuffer.GetRawTextureData<int> ();
		}

		private void loadTextureFromFile (string _fileName) {
			Profiler.BeginSample ("LoadTexture");

			Profiler.BeginSample ("LoadFile");
			byte[] array = cache.LoadTile (zoomLevel, _fileName);
			Profiler.EndSample ();

			Profiler.BeginSample ("LoadImage");
			if (array != null && blockMap.LoadImage (array) && blockMap.height == Constants.MAP_BLOCK_SIZE &&
			    blockMap.width == Constants.MAP_BLOCK_SIZE) {
				Profiler.EndSample ();

				Profiler.EndSample ();
				return;
			}
			Profiler.EndSample ();

			if (array != null) {
				Log.Error ("Map image tile " + _fileName + " has been corrupted, recreating tile");
			}

			if (blockMap.height != Constants.MAP_BLOCK_SIZE || blockMap.width != Constants.MAP_BLOCK_SIZE) {
				blockMap.Resize (Constants.MAP_BLOCK_SIZE, Constants.MAP_BLOCK_SIZE, TextureFormat.ARGB32, false);
			}

			for (int x = 0; x < Constants.MAP_BLOCK_SIZE; x++) {
				for (int y = 0; y < Constants.MAP_BLOCK_SIZE; y++) {
					blockMap.SetPixel (x, y, nullColor);
				}
			}
			Profiler.EndSample ();
		}

		private void saveTextureToFile (string _fileName) {
			Profiler.BeginSample ("SaveTexture");

			Profiler.BeginSample ("EncodePNG");
			byte[] array = blockMap.EncodeToPNG ();
			Profiler.EndSample ();

			cache.SaveTile (zoomLevel, array);
			Profiler.EndSample ();
		}
	}
}