using System;
using System.IO;
using System.Net;
using System.Text;
using HttpListenerRequest = SpaceWizards.HttpListener.HttpListenerRequest;
using HttpListenerResponse = SpaceWizards.HttpListener.HttpListenerResponse;

namespace AllocsFixes.NetConnections.Servers.Web.Handlers {
	public class SessionHandler : AbsHandler {
		private const string pageBasePath = "/";
		private const string steamOpenIdVerifyUrl = "verifysteamopenid";
		private const string steamLoginUrl = "loginsteam";
		
		private readonly string footer = "";
		private readonly string header = "";

		private readonly ConnectionHandler connectionHandler;

		public SessionHandler (string _dataFolder, ConnectionHandler _connectionHandler) : base (null) {
			connectionHandler = _connectionHandler;
			
			if (File.Exists (_dataFolder + "/sessionheader.tmpl")) {
				header = File.ReadAllText (_dataFolder + "/sessionheader.tmpl");
			}

			if (File.Exists (_dataFolder + "/sessionfooter.tmpl")) {
				footer = File.ReadAllText (_dataFolder + "/sessionfooter.tmpl");
			}
		}

		public override void HandleRequest (string _requestPath, HttpListenerRequest _req, HttpListenerResponse _resp, WebConnection _con,
			int _permissionLevel) {
			
			IPEndPoint reqRemoteEndPoint = _req.RemoteEndPoint;
			if (reqRemoteEndPoint == null) {
				_resp.Redirect (pageBasePath);
				return;
			}

			string subpath = _requestPath.Remove (0, urlBasePath.Length);

			StringBuilder result = new StringBuilder ();
			result.Append (header);

			if (subpath.StartsWith (steamOpenIdVerifyUrl)) {
				string remoteEndpointString = reqRemoteEndPoint.ToString ();

				try {
					ulong id = OpenID.Validate (_req);
					if (id > 0) {
						WebConnection con = connectionHandler.LogIn (id, reqRemoteEndPoint.Address);
						int level = GameManager.Instance.adminTools.GetUserPermissionLevel (con.UserId);
						Log.Out ("Steam OpenID login from {0} with ID {1}, permission level {2}",
							remoteEndpointString, con.UserId, level);
						
						Cookie cookie = new Cookie ("sid", con.SessionID, "/") {
							Expired = false,
							Expires = DateTime.MinValue,
							HttpOnly = true,
							Secure = false
						};
						_resp.AppendCookie (cookie);
						_resp.Redirect (pageBasePath);

						return;
					}
				} catch (Exception e) {
					Log.Error ("Error validating login:");
					Log.Exception (e);
				}

				Log.Out ($"Steam OpenID login failed from {remoteEndpointString}");
				result.Append ($"<h1>Login failed, <a href=\"{pageBasePath}\">click to return to main page</a>.</h1>");
			} else if (subpath.StartsWith ("logout")) {
				if (_con != null) {
					connectionHandler.LogOut (_con.SessionID);
					Cookie cookie = new Cookie ("sid", "", "/") {
						Expired = true
					};
					_resp.AppendCookie (cookie);
					_resp.Redirect (pageBasePath);
					return;
				}

				result.Append ($"<h1>Not logged in, <a href=\"{pageBasePath}\">click to return to main page</a>.</h1>");
			} else if (subpath.StartsWith (steamLoginUrl)) {
				string host = (Web.IsSslRedirected (_req) ? "https://" : "http://") + _req.UserHostName;
				string url = OpenID.GetOpenIdLoginUrl (host, host + urlBasePath + steamOpenIdVerifyUrl);
				_resp.Redirect (url);
				return;
			} else {
				result.Append ($"<h1>Unknown command, <a href=\"{pageBasePath}\">click to return to main page</a>.</h1>");
			}

			result.Append (footer);

			_resp.ContentType = MimeType.GetMimeType (".html");
			_resp.ContentEncoding = Encoding.UTF8;
			byte[] buf = Encoding.UTF8.GetBytes (result.ToString ());
			_resp.ContentLength64 = buf.Length;
			_resp.OutputStream.Write (buf, 0, buf.Length);
		}
	}
}