using System.Collections.Generic;
using System.Net;
using AllocsFixes.JSON;
using AllocsFixes.NetConnections.Servers.Web;
using AllocsFixes.NetConnections.Servers.Web.API;

namespace Examples {
	class Markers : AbsRestApi {
		private const int NumRandomMarkers = 5;

		private readonly Dictionary<string, (int, int)> markers = new Dictionary<string, (int, int)> ();

		private static readonly JSONArray emptyResult = new JSONArray ();
		
		public Markers () {
			GameRandom random = GameRandomManager.Instance.CreateGameRandom ();
			
			for (int i = 0; i < NumRandomMarkers; i++) {
				int lat = random.RandomRange (-1000, 1001);
				int lng = random.RandomRange (-1000, 1001);

				markers.Add (WebUtils.GenerateGuid (), (lat, lng));
			}
		}

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;
			
			if (string.IsNullOrEmpty (id)) {
				JSONArray result = new JSONArray ();

				foreach (KeyValuePair<string, (int, int)> kvp in markers) {
					JSONObject marker = new JSONObject ();
					marker.Add ("id", new JSONString (kvp.Key));
					marker.Add ("lat", new JSONNumber (kvp.Value.Item1));
					marker.Add ("lng", new JSONNumber (kvp.Value.Item2));
					result.Add (marker);
				}
				
				SendEnvelopedResult (_context, result);
				return;
			}

			if (!markers.TryGetValue (id, out (int, int) location)) {
				SendEnvelopedResult (_context, emptyResult, HttpStatusCode.NotFound);
				return;
			}

			{
				JSONArray result = new JSONArray ();
				JSONObject marker = new JSONObject ();
				marker.Add ("id", new JSONString (id));
				marker.Add ("lat", new JSONNumber (location.Item1));
				marker.Add ("lng", new JSONNumber (location.Item2));
				result.Add (marker);
				SendEnvelopedResult (_context, result);
			}
		}

		protected override void HandleRestPost (RequestContext _context, JSONNode _jsonBody) {
			if (!(_jsonBody is JSONObject bodyObject)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "BODY_NOT_OBJECT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lat", out int lat)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lng", out int lng)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LNG");
				return;
			}

			string newId = WebUtils.GenerateGuid ();
			markers.Add (newId, (lat, lng));

			JSONString result = new JSONString (newId);
			SendEnvelopedResult (_context, result);
		}

		protected override void HandleRestPut (RequestContext _context, JSONNode _jsonBody) {
			if (!(_jsonBody is JSONObject bodyObject)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "BODY_NOT_OBJECT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lat", out int lat)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lng", out int lng)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LNG");
				return;
			}

			string id = _context.RequestPath;

			if (!markers.TryGetValue (id, out (int, int) location)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.NotFound, _jsonBody, "ID_NOT_FOUND");
				return;
			}
			
			markers [id] = (lat, lng);

			JSONObject result = new JSONObject ();
			result.Add ("id", new JSONString (id));
			result.Add ("lat", new JSONNumber (lat));
			result.Add ("lng", new JSONNumber (lng));
			SendEnvelopedResult (_context, result);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			string id = _context.RequestPath;

			SendEnvelopedResult (_context, null, markers.Remove (id) ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}
	}
}