using System.Collections.Generic;
using System.Text;

namespace AllocsFixes.JSON {
	public class JsonObject : JsonNode {
		private readonly Dictionary<string, JsonNode> nodes = new Dictionary<string, JsonNode> ();

		public JsonNode this [string _name] {
			get => nodes [_name];
			set => nodes [_name] = value;
		}

		public int Count => nodes.Count;

		public List<string> Keys => new List<string> (nodes.Keys);

		public bool ContainsKey (string _name) {
			return nodes.ContainsKey (_name);
		}

		public bool TryGetValue (string _name, out JsonNode _node) {
			return nodes.TryGetValue (_name, out _node);
		}

		public void Add (string _name, JsonNode _node) {
			nodes.Add (_name, _node);
		}

		public override void ToString (StringBuilder _stringBuilder, bool _prettyPrint = false, int _currentLevel = 0) {
			_stringBuilder.Append ("{");
			if (_prettyPrint) {
				_stringBuilder.Append ('\n');
			}

			foreach ((string key, JsonNode value) in nodes) {
				if (_prettyPrint) {
					_stringBuilder.Append (new string ('\t', _currentLevel + 1));
				}

				_stringBuilder.Append ($"\"{key}\":");
				if (_prettyPrint) {
					_stringBuilder.Append (" ");
				}

				value.ToString (_stringBuilder, _prettyPrint, _currentLevel + 1);
				_stringBuilder.Append (",");
				if (_prettyPrint) {
					_stringBuilder.Append ('\n');
				}
			}

			if (nodes.Count > 0) {
				_stringBuilder.Remove (_stringBuilder.Length - (_prettyPrint ? 2 : 1), 1);
			}

			if (_prettyPrint) {
				_stringBuilder.Append (new string ('\t', _currentLevel));
			}

			_stringBuilder.Append ("}");
		}

		public static JsonObject Parse (string _json, ref int _offset) {
			//Log.Out ("ParseObject enter (" + offset + ")");
			JsonObject obj = new JsonObject ();

			bool nextElemAllowed = true;
			_offset++;
			while (true) {
				Parser.SkipWhitespace (_json, ref _offset);
				switch (_json [_offset]) {
					case '"':
						if (nextElemAllowed) {
							JsonString key = JsonString.Parse (_json, ref _offset);
							Parser.SkipWhitespace (_json, ref _offset);
							if (_json [_offset] != ':') {
								throw new MalformedJsonException (
									"Could not parse object, missing colon (\":\") after key");
							}

							_offset++;
							JsonNode val = Parser.ParseInternal (_json, ref _offset);
							obj.Add (key.GetString (), val);
							nextElemAllowed = false;
						} else {
							throw new MalformedJsonException (
								"Could not parse object, found new key without a separating comma");
						}

						break;
					case ',':
						if (!nextElemAllowed) {
							nextElemAllowed = true;
							_offset++;
						} else {
							throw new MalformedJsonException (
								"Could not parse object, found a comma without a key/value pair first");
						}

						break;
					case '}':
						_offset++;

						//Log.Out ("JSON:Parsed Object: " + obj.ToString ());
						return obj;
				}
			}
		}
	}
}