using System.Collections.Generic;
using System.Net;
using AllocsFixes.JSON;
using Webserver;
using Webserver.WebAPI;

namespace Examples {
	public class Markers : AbsRestApi {
		private const int numRandomMarkers = 5;

		private readonly Dictionary<string, (int, int)> markers = new Dictionary<string, (int, int)> ();

		private static readonly JsonArray emptyResult = new JsonArray ();
		
		public Markers () {
			GameRandom random = GameRandomManager.Instance.CreateGameRandom ();
			
			for (int i = 0; i < numRandomMarkers; i++) {
				int lat = random.RandomRange (-1000, 1001);
				int lng = random.RandomRange (-1000, 1001);

				markers.Add (WebUtils.GenerateGuid (), (lat, lng));
			}
		}

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;
			
			if (string.IsNullOrEmpty (id)) {
				JsonArray result = new JsonArray ();

				foreach ((string markerId, (int, int) coordinates) in markers) {
					JsonObject marker = new JsonObject ();
					marker.Add ("id", new JsonString (markerId));
					marker.Add ("lat", new JsonNumber (coordinates.Item1));
					marker.Add ("lng", new JsonNumber (coordinates.Item2));
					result.Add (marker);
				}
				
				SendEnvelopedResult (_context, result);
				return;
			}

			if (!markers.TryGetValue (id, out (int, int) location)) {
				SendEnvelopedResult (_context, emptyResult, HttpStatusCode.NotFound);
				return;
			}

			{
				JsonArray result = new JsonArray ();
				JsonObject marker = new JsonObject ();
				marker.Add ("id", new JsonString (id));
				marker.Add ("lat", new JsonNumber (location.Item1));
				marker.Add ("lng", new JsonNumber (location.Item2));
				result.Add (marker);
				SendEnvelopedResult (_context, result);
			}
		}

		protected override void HandleRestPost (RequestContext _context, JsonNode _jsonBody) {
			if (!(_jsonBody is JsonObject bodyObject)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "BODY_NOT_OBJECT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lat", out int lat)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lng", out int lng)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LNG");
				return;
			}

			string newId = WebUtils.GenerateGuid ();
			markers.Add (newId, (lat, lng));

			JsonString result = new JsonString (newId);
			SendEnvelopedResult (_context, result, HttpStatusCode.Created);
		}

		protected override void HandleRestPut (RequestContext _context, JsonNode _jsonBody) {
			if (!(_jsonBody is JsonObject bodyObject)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "BODY_NOT_OBJECT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lat", out int lat)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (bodyObject, "lng", out int lng)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.BadRequest, _jsonBody, "NO_OR_INVALID_LNG");
				return;
			}

			string id = _context.RequestPath;

			if (!markers.TryGetValue (id, out _)) {
				SendEnvelopedResult (_context, null, HttpStatusCode.NotFound, _jsonBody, "ID_NOT_FOUND");
				return;
			}
			
			markers [id] = (lat, lng);

			JsonObject result = new JsonObject ();
			result.Add ("id", new JsonString (id));
			result.Add ("lat", new JsonNumber (lat));
			result.Add ("lng", new JsonNumber (lng));
			SendEnvelopedResult (_context, result);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			string id = _context.RequestPath;

			SendEnvelopedResult (_context, null, markers.Remove (id) ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}
	}
}