using System.Collections.Generic;
using System.Net;
using AllocsFixes.JSON;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using HttpListenerRequest = SpaceWizards.HttpListener.HttpListenerRequest;

namespace Webserver.WebAPI {
	[UsedImplicitly]
	public class GetPlayerInventory : AbsWebAPI {
		public override void HandleRequest (RequestContext _context) {
			if (_context.Request.QueryString ["userid"] == null) {
				WebUtils.WriteText (_context.Response, "No user id given", HttpStatusCode.BadRequest);
				return;
			}

			string userIdString = _context.Request.QueryString ["userid"];
			if (!PlatformUserIdentifierAbs.TryFromCombinedString (userIdString, out PlatformUserIdentifierAbs userId)) {
				WebUtils.WriteText (_context.Response, "Invalid user id given", HttpStatusCode.BadRequest);
				return;
			}

			Player p = PersistentContainer.Instance.Players [userId, false];
			if (p == null) {
				WebUtils.WriteText (_context.Response, "Unknown user id given", HttpStatusCode.NotFound);
				return;
			}

			GetInventoryArguments (_context.Request, out bool showIconColor, out bool showIconName);

			JsonObject result = DoPlayer (userIdString, p, showIconColor, showIconName);

			WebUtils.WriteJson (_context.Response, result);
		}

		internal static void GetInventoryArguments (HttpListenerRequest _req, out bool _showIconColor, out bool _showIconName) {
			if (_req.QueryString ["showiconcolor"] == null || !bool.TryParse (_req.QueryString ["showiconcolor"], out _showIconColor)) {
				_showIconColor = true;
			}
			
			if (_req.QueryString ["showiconname"] == null || !bool.TryParse (_req.QueryString ["showiconname"], out _showIconName)) {
				_showIconName = true;
			}
		}

		internal static JsonObject DoPlayer (string _steamId, Player _player, bool _showIconColor, bool _showIconName) {
			AllocsFixes.PersistentData.Inventory inv = _player.Inventory;

			JsonObject result = new JsonObject ();

			JsonArray bag = new JsonArray ();
			JsonArray belt = new JsonArray ();
			JsonObject equipment = new JsonObject ();
			result.Add ("userid", new JsonString (_steamId));
			result.Add ("entityid", new JsonNumber (_player.EntityID));
			result.Add ("playername", new JsonString (_player.Name));
			result.Add ("bag", bag);
			result.Add ("belt", belt);
			result.Add ("equipment", equipment);

			DoInventory (belt, inv.belt, _showIconColor, _showIconName);
			DoInventory (bag, inv.bag, _showIconColor, _showIconName);

			AddEquipment (equipment, "head", inv.equipment, EquipmentSlots.Headgear, _showIconColor, _showIconName);
			AddEquipment (equipment, "eyes", inv.equipment, EquipmentSlots.Eyewear, _showIconColor, _showIconName);
			AddEquipment (equipment, "face", inv.equipment, EquipmentSlots.Face, _showIconColor, _showIconName);

			AddEquipment (equipment, "armor", inv.equipment, EquipmentSlots.ChestArmor, _showIconColor, _showIconName);
			AddEquipment (equipment, "jacket", inv.equipment, EquipmentSlots.Jacket, _showIconColor, _showIconName);
			AddEquipment (equipment, "shirt", inv.equipment, EquipmentSlots.Shirt, _showIconColor, _showIconName);

			AddEquipment (equipment, "legarmor", inv.equipment, EquipmentSlots.LegArmor, _showIconColor, _showIconName);
			AddEquipment (equipment, "pants", inv.equipment, EquipmentSlots.Legs, _showIconColor, _showIconName);
			AddEquipment (equipment, "boots", inv.equipment, EquipmentSlots.Feet, _showIconColor, _showIconName);

			AddEquipment (equipment, "gloves", inv.equipment, EquipmentSlots.Hands, _showIconColor, _showIconName);

			return result;
		}

		private static void DoInventory (JsonArray _jsonRes, List<InvItem> _inv, bool _showIconColor, bool _showIconName) {
			for (int i = 0; i < _inv.Count; i++) {
				_jsonRes.Add (GetJsonForItem (_inv [i], _showIconColor, _showIconName));
			}
		}

		private static void AddEquipment (JsonObject _eq, string _slotname, InvItem[] _items, EquipmentSlots _slot, bool _showIconColor, bool _showIconName) {
			int[] slotindices = XUiM_PlayerEquipment.GetSlotIndicesByEquipmentSlot (_slot);

			for (int i = 0; i < slotindices.Length; i++) {
				if (_items? [slotindices [i]] == null) {
					continue;
				}

				InvItem item = _items [slotindices [i]];
				_eq.Add (_slotname, GetJsonForItem (item, _showIconColor, _showIconName));
				return;
			}

			_eq.Add (_slotname, new JsonNull ());
		}

		private static JsonNode GetJsonForItem (InvItem _item, bool _showIconColor, bool _showIconName) {
			if (_item == null) {
				return new JsonNull ();
			}

			JsonObject jsonItem = new JsonObject ();
			jsonItem.Add ("count", new JsonNumber (_item.count));
			jsonItem.Add ("name", new JsonString (_item.itemName));
			
			if (_showIconName) {
				jsonItem.Add ("icon", new JsonString (_item.icon));
			}

			if (_showIconColor) {
				jsonItem.Add ("iconcolor", new JsonString (_item.iconcolor));
			}

			jsonItem.Add ("quality", new JsonNumber (_item.quality));
			if (_item.quality >= 0) {
				jsonItem.Add ("qualitycolor", new JsonString (QualityInfo.GetQualityColorHex (_item.quality)));
			}

			return jsonItem;

		}
	}
}