using System.Collections.Generic;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using Platform.Steam;

namespace AllocsFixes {
	[UsedImplicitly]
	public class ModApi : IModApi {
		public void InitMod (Mod _modInstance) {
			ModEvents.GameStartDone.RegisterHandler (GameAwake);
			ModEvents.SavePlayerData.RegisterHandler (SavePlayerData);
			ModEvents.PlayerSpawning.RegisterHandler (PlayerSpawning);
			ModEvents.PlayerDisconnected.RegisterHandler (PlayerDisconnected);
			ModEvents.PlayerSpawnedInWorld.RegisterHandler (PlayerSpawned);
			ModEvents.ChatMessage.RegisterHandler (ChatMessage);
		}

		private void GameAwake () {
			PersistentContainer.Load ();
		}

		private void SavePlayerData (ClientInfo _cInfo, PlayerDataFile _playerDataFile) {
			if (_cInfo == null) {
				return;
			}

			PersistentContainer.Instance.Players [_cInfo.InternalId, true].Update (_playerDataFile);
		}

		private void PlayerSpawning (ClientInfo _cInfo, int _chunkViewDim, PlayerProfile _playerProfile) {
			if (_cInfo == null) {
				return;
			}

			string owner = null;
			if (_cInfo.PlatformId is UserIdentifierSteam identifierSteam) {
				owner = identifierSteam.OwnerId.ToString ();
			}

			Log.Out (
				$"Player connected, entityid={_cInfo.entityId}, name={_cInfo.playerName}, pltfmid={_cInfo.PlatformId?.CombinedString ?? "<unknown>"}, crossid={_cInfo.CrossplatformId?.CombinedString ?? "<unknown/none>"}, steamOwner={owner ?? "<unknown/none>"}, ip={_cInfo.ip}"
			);
		}

		private void PlayerDisconnected (ClientInfo _cInfo, bool _bShutdown) {
			if (_cInfo == null) {
				return;
			}

			Player p = PersistentContainer.Instance.Players [_cInfo.InternalId, false];
			if (p != null) {
				p.SetOffline ();
			} else {
				Log.Out ("Disconnected player not found in client list...");
			}

			PersistentContainer.Instance.Save ();
		}

		private void PlayerSpawned (ClientInfo _cInfo, RespawnType _respawnReason, Vector3i _spawnPos) {
			if (_cInfo == null) {
				return;
			}

			PersistentContainer.Instance.Players [_cInfo.InternalId, true].SetOnline (_cInfo);
			PersistentContainer.Instance.Save ();
		}

		private const string testChatAnswer =
			"     [ff0000]I[-] [ff7f00]W[-][ffff00]A[-][80ff00]S[-] [00ffff]H[-][0080ff]E[-][0000ff]R[-][8b00ff]E[-]";

		private bool ChatMessage (ClientInfo _cInfo, EChatType _type, int _senderId, string _msg, string _mainName,
			bool _localizeMain, List<int> _recipientEntityIds) {
			if (_cInfo == null) {
				return true;
			}

			if (string.IsNullOrEmpty (_msg) || !_msg.EqualsCaseInsensitive ("/alloc")) {
				return true;
			}

			Log.Out ($"Sent chat hook reply to {_cInfo.InternalId}");
			_cInfo.SendPackage (NetPackageManager.GetPackage<NetPackageChat> ().Setup (EChatType.Whisper, -1, testChatAnswer, "", false, null));

			return false;
		}
	}
}