using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using Platform.Steam;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.UrlHandlers {
	public class SessionHandler : AbsHandler {
		private const string pageBasePath = "/app";
		private const string pageErrorPath = "/app/error/";

		private const string steamOpenIdVerifyUrl = "verifysteamopenid";
		private const string steamLoginUrl = "loginsteam";
		private const string steamLoginFailedPage = "SteamLoginFailed";

		private const string userPassLoginUrl = "login";
		public const string userPassLoginName = "User/pass";
		public const string userPassErrorPage = "UserPassLoginFailed";

		private readonly ConnectionHandler connectionHandler;

		public SessionHandler (ConnectionHandler _connectionHandler) : base (null) {
			connectionHandler = _connectionHandler;
		}

		public override void HandleRequest (RequestContext _context) {
			if (_context.Request.RemoteEndPoint == null) {
				_context.Response.Redirect (pageErrorPath + "NoRemoteEndpoint");
				return;
			}

			string subpath = _context.RequestPath.Remove (0, urlBasePath.Length);

			string remoteEndpointString = _context.Request.RemoteEndPoint!.ToString ();

			if (subpath.StartsWith (steamOpenIdVerifyUrl)) {
				HandleSteamVerification (_context, remoteEndpointString);
				return;
			}

			if (subpath.StartsWith ("logout")) {
				HandleLogout (_context);
				return;
			}

			if (subpath.StartsWith (steamLoginUrl)) {
				HandleSteamLogin (_context);
				return;
			}

			if (subpath.StartsWith (userPassLoginUrl)) {
				HandleUserPassLogin (_context, remoteEndpointString);
				return;
			}

			_context.Response.Redirect (pageErrorPath + "InvalidSessionsCommand");
		}

		private void HandleUserPassLogin (RequestContext _context, string _remoteEndpointString) {
			if (!_context.Request.HasEntityBody) {
				_context.Response.Redirect (pageErrorPath + "NoLoginData");
				return;
			}

			Stream requestInputStream = _context.Request.InputStream;

			byte[] jsonInputData = new byte[_context.Request.ContentLength64];
			requestInputStream.Read (jsonInputData, 0, (int)_context.Request.ContentLength64);

			IDictionary<string, object> inputJson;
			try {
				inputJson = JsonSerializer.Deserialize<IDictionary<string, object>> (jsonInputData);
			} catch (Exception e) {
				Log.Error ("Error deserializing JSON from user/password login:");
				Log.Exception (e);
				_context.Response.Redirect (pageErrorPath + "InvalidLoginJson");
				return;
			}

			if (!inputJson.TryGetValue ("username", out object fieldNode) || fieldNode is not string username) {
				_context.Response.Redirect (pageErrorPath + "InvalidLoginJson");
				return;
			}

			if (!inputJson.TryGetValue ("password", out fieldNode) || fieldNode is not string password) {
				_context.Response.Redirect (pageErrorPath + "InvalidLoginJson");
				return;
			}

			AdminWebUsers.WebUser? webUser = AdminWebUsers.Instance.GetUser (username, password);

			if (!webUser.HasValue) {
				Log.Out ($"[Web] User/pass login failed from {_remoteEndpointString}");
				_context.Response.Redirect (pageErrorPath + "UserPassInvalid");
				return;
			}

			HandleUserIdLogin (connectionHandler, _context, _remoteEndpointString, userPassLoginName, userPassErrorPage, webUser.Value.Name, webUser.Value.PlatformUser, webUser.Value.CrossPlatformUser);
		}

		private void HandleSteamLogin (RequestContext _context) {
			string host = $"{(WebUtils.IsSslRedirected (_context.Request) ? "https://" : "http://")}{_context.Request.UserHostName}";
			string url = OpenID.GetOpenIdLoginUrl (host, $"{host}{urlBasePath}{steamOpenIdVerifyUrl}");
			_context.Response.Redirect (url);
		}

		private void HandleLogout (RequestContext _context) {
			Cookie cookie = new Cookie ("sid", "", "/") {
				Expired = true
			};
			_context.Response.AppendCookie (cookie);

			if (_context.Connection == null) {
				_context.Response.Redirect (pageErrorPath + "NotLoggedIn");
				return;
			}

			connectionHandler.LogOut (_context.Connection.SessionID);
			_context.Response.Redirect (pageBasePath);
		}

		private void HandleSteamVerification (RequestContext _context, string _remoteEndpointString) {
			ulong id;
			try {
				id = OpenID.Validate (_context.Request);
			} catch (Exception e) {
				Log.Error ($"[Web] Error validating Steam login from {_remoteEndpointString}:");
				Log.Exception (e);
				_context.Response.Redirect (pageErrorPath + steamLoginFailedPage);
				return;
			}

			if (id <= 0) {
				Log.Out ($"[Web] Steam OpenID login failed (invalid ID) from {_remoteEndpointString}");
				_context.Response.Redirect (pageErrorPath + steamLoginFailedPage);
				return;
			}

			UserIdentifierSteam userId = new UserIdentifierSteam (id);
			HandleUserIdLogin (connectionHandler, _context, _remoteEndpointString, "Steam OpenID", steamLoginFailedPage, userId.ToString (), userId);
		}

		public static void HandleUserIdLogin (ConnectionHandler _connectionHandler, RequestContext _context, string _remoteEndpointString,
			string _loginName, string _errorPage, string _username, PlatformUserIdentifierAbs _userId, PlatformUserIdentifierAbs _crossUserId = null, bool _redirect = true) {
			try {
				WebConnection con = _connectionHandler.LogIn (_context.Request.RemoteEndPoint!.Address, _username, _userId, _crossUserId);

				int level1 = GameManager.Instance.adminTools.Users.GetUserPermissionLevel (_userId);
				int level2 = int.MaxValue;
				if (_crossUserId != null) {
					level2 = GameManager.Instance.adminTools.Users.GetUserPermissionLevel (_crossUserId);
				}

				int higherLevel = Math.Min (level1, level2);

				Log.Out ($"[Web] {_loginName} login from {_remoteEndpointString}, name {_username} with ID {_userId}, CID {(_crossUserId != null ? _crossUserId : "none")}, permission level {higherLevel}");
				Cookie cookie = new Cookie ("sid", con.SessionID, "/") {
					Expired = false,
					Expires = DateTime.MinValue,
					HttpOnly = true,
					Secure = false
				};
				_context.Response.AppendCookie (cookie);

				if (_redirect) {
					_context.Response.Redirect (pageBasePath);
				}
			} catch (Exception e) {
				Log.Error ($"[Web] Error during {_loginName} login:");
				Log.Exception (e);
				if (_redirect) {
					_context.Response.Redirect (pageErrorPath + _errorPage);
				}
			}
		}
	}
}