using System.Collections.Generic;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.UrlHandlers {
	public class UserStatusHandler : AbsHandler {
		public UserStatusHandler (string _moduleName = null) : base (_moduleName) {
		}

		private static readonly byte[] jsonLoggedInKey = JsonWriter.GetEncodedPropertyNameWithBeginObject ("loggedIn");
		private static readonly byte[] jsonUsernameKey = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("username");
		private static readonly byte[] jsonPermissionLevelKey = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("permissionLevel");
		private static readonly byte[] jsonPermissionsKey = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("permissions");

		private static readonly byte[] jsonModuleKey = JsonWriter.GetEncodedPropertyNameWithBeginObject ("module");
		private static readonly byte[] jsonAllowedKey = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("allowed");

		private static readonly byte[][] jsonMethodNameKeys;

		static UserStatusHandler () {
			jsonMethodNameKeys = new byte[(int)ERequestMethod.Count][];
			for (int i = 0; i < jsonMethodNameKeys.Length; i++) {
				ERequestMethod method = (ERequestMethod)i;
				jsonMethodNameKeys [i] = JsonWriter.GetEncodedPropertyName (method.ToStringCached ());
			}
		}

		public override void HandleRequest (RequestContext _context) {
			WebUtils.PrepareEnvelopedResult (out JsonWriter writer);
			
			writer.WriteRaw (jsonLoggedInKey);
			writer.WriteBoolean (_context.Connection != null);
			
			writer.WriteRaw (jsonUsernameKey);
			writer.WriteString (_context.Connection != null ? _context.Connection.Username : string.Empty);
			
			writer.WriteRaw (jsonPermissionLevelKey);
			writer.WriteInt32 (_context.PermissionLevel);
			
			writer.WriteRaw (jsonPermissionsKey);
			writer.WriteBeginArray ();

			List<AdminWebModules.WebModule> list = AdminWebModules.Instance.GetModules ();
			for (int iModule = 0; iModule < list.Count; iModule++) {
				AdminWebModules.WebModule perm = list [iModule];
				
				if (iModule > 0) {
					writer.WriteValueSeparator ();
				}

				writer.WriteRaw (jsonModuleKey);
				writer.WriteString (perm.Name);

				writer.WriteRaw (jsonAllowedKey);
				
				writer.WriteBeginObject ();

				if (perm.LevelPerMethod == null) {
					writer.WriteRaw (jsonMethodNameKeys [(int)ERequestMethod.GET]);
					writer.WriteBoolean (perm.LevelGlobal >= _context.PermissionLevel);
				} else {
					bool first = true;
					for (int iMethod = 0; iMethod < perm.LevelPerMethod.Length; iMethod++) {
						int methodLevel = perm.LevelPerMethod [iMethod];
						
						if (methodLevel == AdminWebModules.MethodLevelNotSupported) {
							continue;
						}

						if (methodLevel == AdminWebModules.MethodLevelInheritGlobal) {
							methodLevel = perm.LevelGlobal;
						}

						if (!first) {
							writer.WriteValueSeparator ();
						}

						first = false;
						
						writer.WriteRaw (jsonMethodNameKeys [iMethod]);
						writer.WriteBoolean (methodLevel >= _context.PermissionLevel);
					}
				}

				writer.WriteEndObject ();
				

				writer.WriteEndObject ();
			}

			writer.WriteEndArray ();
			
			writer.WriteEndObject ();
			
			WebUtils.SendEnvelopedResult (_context, ref writer);
		}
	}
}