using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.Permissions;
using Webserver.WebAPI;

namespace Examples {
	[UsedImplicitly]
	public class Markers : AbsRestApi {
		private const int numRandomMarkers = 5;

		private const string defaultIcon =
			"https://upload.wikimedia.org/wikipedia/commons/thumb/1/11/Blue_question_mark_icon.svg/1200px-Blue_question_mark_icon.svg.png";
		
		private readonly Dictionary<string, (int, int, string)> markers = new Dictionary<string, (int, int, string)> ();

		public Markers () {
			GameRandom random = GameRandomManager.Instance.CreateGameRandom ();
			
			for (int i = 0; i < numRandomMarkers; i++) {
				int lat = random.RandomRange (-1000, 1001);
				int lng = random.RandomRange (-1000, 1001);

				markers.Add (WebUtils.GenerateGuid (), (lat, lng, null));
			}
		}

		private static readonly byte[] jsonKeyId = JsonWriter.GetEncodedPropertyNameWithBeginObject ("id");
		private static readonly byte[] jsonKeyLat = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("lat");
		private static readonly byte[] jsonKeyLng = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("lng");
		private static readonly byte[] jsonKeyIcon = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("icon");

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;
			
			PrepareEnvelopedResult (out JsonWriter writer);
			
			if (string.IsNullOrEmpty (id)) {
				writer.WriteBeginArray ();

				bool first = true;
				foreach ((string markerId, (int, int, string) properties) in markers) {
					if (!first) {
						writer.WriteValueSeparator ();
					}
					
					first = false;
					
					writeMarkerJson (ref writer, markerId, properties);
				}

				writer.WriteEndArray ();
				SendEnvelopedResult (_context, ref writer);
				return;
			}

			if (!markers.TryGetValue (id, out (int, int, string) properties2)) {
				writer.WriteRaw (JsonEmptyData);
				SendEnvelopedResult (_context, ref writer, HttpStatusCode.NotFound);
				return;
			}

			{
				writer.WriteBeginArray ();
				
				writeMarkerJson (ref writer, id, properties2);
				
				writer.WriteEndArray ();
				SendEnvelopedResult (_context, ref writer);
			}
		}

		private void writeMarkerJson (ref JsonWriter _writer, string _markerId, (int, int, string) _properties) {
			_writer.WriteRaw (jsonKeyId);
			_writer.WriteString (_markerId);
			_writer.WriteRaw (jsonKeyLat);
			_writer.WriteInt32 (_properties.Item1);
			_writer.WriteRaw (jsonKeyLng);
			_writer.WriteInt32 (_properties.Item2);
			_writer.WriteRaw (jsonKeyIcon);
			_writer.WriteString (_properties.Item3 ?? defaultIcon);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!TryGetJsonField (_jsonInput, "lat", out int lat)) {
				SendErrorResult (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (_jsonInput, "lng", out int lng)) {
				SendErrorResult (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_OR_INVALID_LNG");
				return;
			}

			TryGetJsonField (_jsonInput, "icon", out string icon);
			if (string.IsNullOrEmpty (icon)) {
				icon = null;
			}

			string newId = WebUtils.GenerateGuid ();
			markers.Add (newId, (lat, lng, icon));

			PrepareEnvelopedResult (out JsonWriter writer);
			writer.WriteString (newId);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.Created);
		}

		protected override void HandleRestPut (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!TryGetJsonField (_jsonInput, "lat", out int lat)) {
				SendErrorResult (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_OR_INVALID_LAT");
				return;
			}

			if (!TryGetJsonField (_jsonInput, "lng", out int lng)) {
				SendErrorResult (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_OR_INVALID_LNG");
				return;
			}

			bool keepIcon = !_jsonInput.TryGetValue ("icon", out _);
			TryGetJsonField (_jsonInput, "icon", out string icon);
			if (string.IsNullOrEmpty (icon)) {
				icon = null;
			}

			string id = _context.RequestPath;

			if (!markers.TryGetValue (id, out (int, int, string) properties)) {
				SendErrorResult (_context, HttpStatusCode.NotFound, _jsonInputData, "ID_NOT_FOUND");
				return;
			}

			if (keepIcon) {
				icon = properties.Item3;
			}
			markers [id] = (lat, lng, icon);

			PrepareEnvelopedResult (out JsonWriter writer);
			writer.WriteRaw (jsonKeyId);
			writer.WriteString (id);
			writer.WriteRaw (jsonKeyLat);
			writer.WriteInt32 (lat);
			writer.WriteRaw (jsonKeyLng);
			writer.WriteInt32 (lng);
			writer.WriteRaw (jsonKeyIcon);
			writer.WriteString (icon);
			writer.WriteEndObject ();
			SendEnvelopedResult (_context, ref writer);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);
			writer.WriteRaw (JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, markers.Remove (id) ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}
		
		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.PermissionLevelGuest,
			AdminWebModules.PermissionLevelUser,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}