using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class WebModules : AbsRestApi {
		private const string propertyModule = "module";
		private const string propertyPermissionLevelGlobal = "permissionLevelGlobal";
		private const string propertyPermissionLevelPerMethod = "permissionLevelPerMethod";
		private const string propertyIsDefault = "isDefault";
		

		private static readonly byte[] jsonKeyModule = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyModule);
		private static readonly byte[] jsonKeyPermissionLevelGlobal = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevelGlobal);
		private static readonly byte[] jsonKeyPermissionLevelPerMethod = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevelPerMethod);
		private static readonly byte[] jsonKeyIsDefault = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyIsDefault);

		private static readonly byte[][] jsonMethodNameKeys;

		static WebModules () {
			jsonMethodNameKeys = new byte[(int)ERequestMethod.Count][];
			for (int i = 0; i < jsonMethodNameKeys.Length; i++) {
				ERequestMethod method = (ERequestMethod)i;
				jsonMethodNameKeys [i] = JsonWriter.GetEncodedPropertyName (method.ToStringCached ());
			}
		}

		private static AdminWebModules ModulesInstance => AdminWebModules.Instance;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);
			
			if (string.IsNullOrEmpty (id)) {
				
				writer.WriteBeginArray ();
			
				bool first = true;
				foreach (AdminWebModules.WebModule module in ModulesInstance.GetModules ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeModuleJson (ref writer, module);
				}

				writer.WriteEndArray ();
				
				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeModuleJson (ref JsonWriter _writer, AdminWebModules.WebModule _module) {
			_writer.WriteRaw (jsonKeyModule);
			_writer.WriteString (_module.Name);
			_writer.WriteRaw (jsonKeyPermissionLevelGlobal);
			_writer.WriteInt32 (_module.LevelGlobal);
			_writer.WriteRaw (jsonKeyPermissionLevelPerMethod);
			
			_writer.WriteBeginObject ();

			if (_module.LevelPerMethod != null) {
				bool first = true;
				for (int iMethod = 0; iMethod < _module.LevelPerMethod.Length; iMethod++) {
					int methodLevel = _module.LevelPerMethod [iMethod];
						
					if (methodLevel == AdminWebModules.MethodLevelNotSupported) {
						continue;
					}

					if (!first) {
						_writer.WriteValueSeparator ();
					}

					first = false;
						
					_writer.WriteRaw (jsonMethodNameKeys [iMethod]);
					if (methodLevel == AdminWebModules.MethodLevelInheritGlobal) {
						_writer.WriteString ("inherit");
					} else {
						_writer.WriteInt32 (methodLevel);
					}
				}
			}

			_writer.WriteEndObject ();
			
			_writer.WriteRaw (jsonKeyIsDefault);
			_writer.WriteBoolean (_module.IsDefault);
			
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			string id = _context.RequestPath;

			// TODO
			
			if (string.IsNullOrEmpty (id)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_MODULE");
				return;
			}

			if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPermissionLevelGlobal, out int permissionLevel)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, "NO_OR_INVALID_PERMISSION_LEVEL");
				return;
			}

			// ModulesInstance.AddModule (id, permissionLevel);

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			string id = _context.RequestPath;

			bool removed = ModulesInstance.RemoveModule (id);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}