using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class Whitelist : AbsRestApi {
		private const string propertyName = "name";
		private const string propertyUserId = "userId";
		private const string propertyGroupId = "groupId";

		private static readonly byte[] jsonKeyUsers = JsonWriter.GetEncodedPropertyNameWithBeginObject ("users");
		private static readonly byte[] jsonKeyGroups = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("groups");

		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyName);
		private static readonly byte[] jsonKeyUserId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyUserId);

		private static readonly byte[] jsonKeyGroupId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyGroupId);

		private static AdminWhitelist WhitelistInstance => GameManager.Instance.adminTools.Whitelist;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);

			if (string.IsNullOrEmpty (id)) {
				writer.WriteRaw (jsonKeyUsers);
				writer.WriteBeginArray ();

				bool first = true;
				foreach ((_, AdminWhitelist.WhitelistUser user) in WhitelistInstance.GetUsers ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeUserJson (ref writer, user);
				}

				writer.WriteEndArray ();

				writer.WriteRaw (jsonKeyGroups);
				writer.WriteBeginArray ();

				first = true;
				foreach ((_, AdminWhitelist.WhitelistGroup group) in WhitelistInstance.GetGroups ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeGroupJson (ref writer, group);
				}

				writer.WriteEndArray ();

				writer.WriteEndObject ();

				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeUserJson (ref JsonWriter _writer, AdminWhitelist.WhitelistUser _userPermission) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_userPermission.Name ?? "");
			_writer.WriteRaw (jsonKeyUserId);
			JsonCommons.WritePlatformUserIdentifier (ref _writer, _userPermission.UserIdentifier);
			_writer.WriteEndObject ();
		}

		private void writeGroupJson (ref JsonWriter _writer, AdminWhitelist.WhitelistGroup _groupPermission) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_groupPermission.Name ?? "");
			_writer.WriteRaw (jsonKeyGroupId);
			_writer.WriteString (_groupPermission.SteamIdGroup);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!PermissionsApiHelpers.TryParseId (_context, _jsonInputData, out PlatformUserIdentifierAbs userId, out string groupId)) {
				return;
			}

			JsonCommons.TryGetJsonField (_jsonInput, propertyName, out string name);

			if (userId != null) {
				WhitelistInstance.AddUser (name, userId);
			} else {
				WhitelistInstance.AddGroup (name, groupId);
			}

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			if (!PermissionsApiHelpers.TryParseId (_context, null, out PlatformUserIdentifierAbs userId, out string groupId)) {
				return;
			}

			bool removed = userId != null ? WhitelistInstance.RemoveUser (userId) : WhitelistInstance.RemoveGroup (groupId);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}