using System.Collections.Generic;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.Permissions;
using Webserver.WebAPI;

namespace AllocsFixes.WebAPIs {
	[UsedImplicitly]
	public class GetPlayersLocation : AbsWebAPI {
		private static readonly byte[] jsonKeySteamId = JsonWriter.GetEncodedPropertyNameWithBeginObject ("steamid");
		private static readonly byte[] jsonKeyCrossPlatformId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("crossplatformid");
		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("name");
		private static readonly byte[] jsonKeyOnline = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("online");
		private static readonly byte[] jsonKeyPosition = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("position");

		public override void HandleRequest (RequestContext _context) {
			AdminTools admTools = GameManager.Instance.adminTools;
			PlatformUserIdentifierAbs userId = _context.Connection?.UserId;

			bool listOffline = false;
			if (_context.Request.QueryString ["offline"] != null) {
				bool.TryParse (_context.Request.QueryString ["offline"], out listOffline);
			}

			bool bViewAll = PermissionUtils.CanViewAllPlayers (_context.PermissionLevel);

			JsonWriter writer = new JsonWriter ();
			
			writer.WriteBeginArray ();
			
			bool first = true;

			Players playersList = PersistentContainer.Instance.Players;

			foreach (KeyValuePair<PlatformUserIdentifierAbs, Player> kvp in playersList.Dict) {
				if (admTools != null) {
					if (admTools.Blacklist.IsBanned (kvp.Key, out _, out _)) {
						continue;
					}
				}

				Player p = kvp.Value;

				if (listOffline || p.IsOnline) {
					if (bViewAll || p.InternalId.Equals (userId)) {

						if (!first) {
							writer.WriteValueSeparator ();
						}

						first = false;
				
						writer.WriteRaw (jsonKeySteamId);
						writer.WriteString (kvp.Value.PlatformId.CombinedString);
				
						writer.WriteRaw (jsonKeyCrossPlatformId);
						writer.WriteString (kvp.Value.CrossPlatformId?.CombinedString ?? "");
				
						writer.WriteRaw (jsonKeyName);
						writer.WriteString (p.Name);
				
						writer.WriteRaw (jsonKeyOnline);
						writer.WriteBoolean (p.IsOnline);
				
						writer.WriteRaw (jsonKeyPosition);
						JsonCommons.WriteVector3I (ref writer, p.LastPosition);
				
						writer.WriteEndObject ();
					}
				}
			}
			
			writer.WriteEndArray ();
			
			WebUtils.WriteJsonData (_context.Response, ref writer);
		}
	}
}