using System.Collections.Generic;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.WebAPI;

namespace AllocsFixes.WebAPIs {
	[UsedImplicitly]
	public class GetPlayersOnline : AbsWebAPI {
		private static readonly byte[] jsonKeySteamId = JsonWriter.GetEncodedPropertyNameWithBeginObject ("steamid");
		private static readonly byte[] jsonKeyCrossPlatformId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("crossplatformid");
		private static readonly byte[] jsonKeyEntityId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("entityid");
		private static readonly byte[] jsonKeyIp = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("ip");
		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("name");
		private static readonly byte[] jsonKeyOnline = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("online");
		private static readonly byte[] jsonKeyPosition = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("position");

		private static readonly byte[] jsonKeyLevel = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("level");
		private static readonly byte[] jsonKeyHealth = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("health");
		private static readonly byte[] jsonKeyStamina = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("stamina");
		private static readonly byte[] jsonKeyZombieKills = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("zombiekills");
		private static readonly byte[] jsonKeyPlayerKills = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("playerkills");
		private static readonly byte[] jsonKeyPlayerDeaths = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("playerdeaths");
		private static readonly byte[] jsonKeyScore = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("score");

		private static readonly byte[] jsonKeyTotalPlaytime = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("totalplaytime");
		private static readonly byte[] jsonKeyLastOnline = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("lastonline");
		private static readonly byte[] jsonKeyPing = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("ping");

		public override void HandleRequest (RequestContext _context) {
			JsonWriter writer = new JsonWriter ();
			
			writer.WriteBeginArray ();
			
			bool first = true;

			World w = GameManager.Instance.World;
			foreach (KeyValuePair<int, EntityPlayer> current in w.Players.dict) {
				ClientInfo ci = ConnectionManager.Instance.Clients.ForEntityId (current.Key);
				Player player = PersistentContainer.Instance.Players.GetByInternalId (ci.InternalId);

				if (!first) {
					writer.WriteValueSeparator ();
				}

				first = false;
				
				writer.WriteRaw (jsonKeySteamId);
				writer.WriteString (ci.PlatformId.CombinedString);
				
				writer.WriteRaw (jsonKeyCrossPlatformId);
				writer.WriteString (ci.CrossplatformId?.CombinedString ?? "");
				
				writer.WriteRaw (jsonKeyEntityId);
				writer.WriteInt32 (ci.entityId);
				
				writer.WriteRaw (jsonKeyIp);
				writer.WriteString (ci.ip);
				
				writer.WriteRaw (jsonKeyName);
				writer.WriteString (current.Value.EntityName);
				
				writer.WriteRaw (jsonKeyOnline);
				writer.WriteBoolean (true);
				
				writer.WriteRaw (jsonKeyPosition);
				JsonCommons.WriteVector3I (ref writer, new Vector3i (current.Value.GetPosition ()));
				
				writer.WriteRaw (jsonKeyLevel);
				writer.WriteSingle (player?.Level ?? -1);
				
				writer.WriteRaw (jsonKeyHealth);
				writer.WriteInt32 (current.Value.Health);
				
				writer.WriteRaw (jsonKeyStamina);
				writer.WriteSingle (current.Value.Stamina);
				
				writer.WriteRaw (jsonKeyZombieKills);
				writer.WriteInt32 (current.Value.KilledZombies);
				
				writer.WriteRaw (jsonKeyPlayerKills);
				writer.WriteInt32 (current.Value.KilledPlayers);
				
				writer.WriteRaw (jsonKeyPlayerDeaths);
				writer.WriteInt32 (current.Value.Died);
				
				writer.WriteRaw (jsonKeyScore);
				writer.WriteInt32 (current.Value.Score);

				writer.WriteRaw (jsonKeyTotalPlaytime);
				writer.WriteInt64 (player?.TotalPlayTime ?? -1);
				
				writer.WriteRaw (jsonKeyLastOnline);
				writer.WriteString (player != null ? player.LastOnline.ToString ("s") : string.Empty);

				writer.WriteRaw (jsonKeyPing);
				writer.WriteInt32 (ci.ping);
				
				writer.WriteEndObject ();
			}
			
			writer.WriteEndArray ();
			
			WebUtils.WriteJsonData (_context.Response, ref writer);
		}
	}
}