using System.Collections.Generic;
using System.Net;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.Permissions;
using Webserver.WebAPI;

namespace AllocsFixes.WebAPIs {
	[UsedImplicitly]
	public class GetLandClaims : AbsWebAPI {
		private static readonly byte[] jsonKeyClaimSize = JsonWriter.GetEncodedPropertyNameWithBeginObject ("claimsize");
		private static readonly byte[] jsonKeyClaimOwners = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("claimowners");

		private static readonly byte[] jsonKeySteamId = JsonWriter.GetEncodedPropertyNameWithBeginObject ("steamid");
		private static readonly byte[] jsonKeyCrossPlatformId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("crossplatformid");
		private static readonly byte[] jsonKeyClaimActive = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("claimactive");
		private static readonly byte[] jsonKeyPlayerName = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("playername");
		private static readonly byte[] jsonKeyClaims = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("claims");

		public override void HandleRequest (RequestContext _context) {
			PlatformUserIdentifierAbs requestedUserId = null;
			if (_context.Request.QueryString ["userid"] != null) {
				if (!PlatformUserIdentifierAbs.TryFromCombinedString (_context.Request.QueryString ["userid"], out requestedUserId)) {
					WebUtils.WriteText (_context.Response, "Invalid user id given", HttpStatusCode.BadRequest);
					return;
				}
			}

			// default user, cheap way to avoid 'null reference exception'
			PlatformUserIdentifierAbs userId = _context.Connection?.UserId;

			bool bViewAll = PermissionUtils.CanViewAllClaims (_context.PermissionLevel);

			LandClaimList.OwnerFilter[] ownerFilters = null;
			if (requestedUserId != null || !bViewAll) {
				if (requestedUserId != null && !bViewAll) {
					ownerFilters = new[] {
						LandClaimList.UserIdFilter (userId),
						LandClaimList.UserIdFilter (requestedUserId),
					};
				} else if (!bViewAll) {
					ownerFilters = new[] {LandClaimList.UserIdFilter (userId)};
				} else {
					ownerFilters = new[] {LandClaimList.UserIdFilter (requestedUserId)};
				}
			}

			LandClaimList.PositionFilter[] posFilters = null;

			Dictionary<Player, List<Vector3i>> claims = LandClaimList.GetLandClaims (ownerFilters, posFilters);

			JsonWriter writer = new JsonWriter ();
			
			writer.WriteRaw (jsonKeyClaimSize);
			writer.WriteInt32 (GamePrefs.GetInt (EnumUtils.Parse<EnumGamePrefs> (nameof(EnumGamePrefs.LandClaimSize))));
			
			writer.WriteRaw (jsonKeyClaimOwners);
			writer.WriteBeginArray ();
			
			bool first = true;

			foreach (KeyValuePair<Player, List<Vector3i>> kvp in claims) {
				if (!first) {
					writer.WriteValueSeparator ();
				}

				first = false;
				
				writer.WriteRaw (jsonKeySteamId);
				writer.WriteString (kvp.Key.PlatformId?.CombinedString ?? "");
				
				writer.WriteRaw (jsonKeyCrossPlatformId);
				writer.WriteString (kvp.Key.CrossPlatformId?.CombinedString ?? "");
				
				writer.WriteRaw (jsonKeyClaimActive);
				writer.WriteBoolean (kvp.Key.LandProtectionActive);
				
				writer.WriteRaw (jsonKeyPlayerName);
				if (kvp.Key.Name.Length > 0) {
					writer.WriteString (kvp.Key.Name);
				} else {
					writer.WriteNull ();
				}
				
				writer.WriteRaw (jsonKeyClaims);
				writer.WriteBeginArray ();

				bool first2 = true;

				foreach (Vector3i v in kvp.Value) {
					if (!first2) {
						writer.WriteValueSeparator ();
					}

					first2 = false;
					
					JsonCommons.WriteVector3I (ref writer, v);
				}
				
				writer.WriteEndArray ();
				
				writer.WriteEndObject ();
			}

			writer.WriteEndArray ();
			
			writer.WriteEndObject ();
			WebUtils.WriteJsonData (_context.Response, ref writer);
		}
	}
}