using System.Collections.Generic;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.WebAPI;

namespace AllocsFixes.WebAPIs {
	[UsedImplicitly]
	public class GetLog : AbsWebAPI {
		private const int MAX_COUNT = 1000;
		
		private static readonly byte[] jsonKeyFirstLine = JsonWriter.GetEncodedPropertyNameWithBeginObject ("firstLine");
		private static readonly byte[] jsonKeyLastLine = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("lastLine");
		private static readonly byte[] jsonKeyEntries = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("entries");

		private static readonly byte[] jsonKeyEntDate = JsonWriter.GetEncodedPropertyNameWithBeginObject ("date");
		private static readonly byte[] jsonKeyEntTime = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("time");
		private static readonly byte[] jsonKeyEntUptime = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("uptime");
		private static readonly byte[] jsonKeyEntMsg = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("msg");
		private static readonly byte[] jsonKeyEntTrace = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("trace");
		private static readonly byte[] jsonKeyEntType = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("type");

		public override void HandleRequest (RequestContext _context) {
			if (_context.Request.QueryString ["count"] == null || !int.TryParse (_context.Request.QueryString ["count"], out var count)) {
				count = 50;
			}

			if (count == 0) {
				count = 1;
			}

			if (count > MAX_COUNT) {
				count = MAX_COUNT;
			}

			if (count < -MAX_COUNT) {
				count = -MAX_COUNT;
			}

			if (_context.Request.QueryString ["firstLine"] == null || !int.TryParse (_context.Request.QueryString ["firstLine"], out var firstLine)) {
				firstLine = count > 0 ? LogBuffer.Instance.OldestLine : LogBuffer.Instance.LatestLine;
			}

			List<LogBuffer.LogEntry> logEntries = LogBuffer.Instance.GetRange (ref firstLine, count, out var lastLine);

			JsonWriter writer = new JsonWriter ();
			
			writer.WriteRaw (jsonKeyFirstLine);
			writer.WriteInt32 (firstLine);
			
			writer.WriteRaw (jsonKeyLastLine);
			writer.WriteInt32 (lastLine);
			
			writer.WriteRaw (jsonKeyEntries);
			writer.WriteBeginArray ();
			
			bool first = true;

			foreach (LogBuffer.LogEntry logEntry in logEntries) {
				if (!first) {
					writer.WriteValueSeparator ();
				}

				first = false;
				
				var logEntryTimestamp = logEntry.Timestamp;

				writer.WriteRaw (jsonKeyEntDate);
				writer.WriteString ($"{logEntryTimestamp.Year:0000}-{logEntryTimestamp.Month:00}-{logEntryTimestamp.Day:00}");
				
				writer.WriteRaw (jsonKeyEntTime);
				writer.WriteString ($"{logEntryTimestamp.Hour:00}:{logEntryTimestamp.Minute:00}:{logEntryTimestamp.Second:00}");
				
				writer.WriteRaw (jsonKeyEntUptime);
				writer.WriteString (logEntry.Uptime.ToString());
				
				writer.WriteRaw (jsonKeyEntMsg);
				writer.WriteString (logEntry.Message);
				
				writer.WriteRaw (jsonKeyEntTrace);
				writer.WriteString (logEntry.Trace);
				
				writer.WriteRaw (jsonKeyEntType);
				writer.WriteString (logEntry.Type.ToStringCached ());

				writer.WriteEndObject ();
			}

			writer.WriteEndArray ();
			
			writer.WriteEndObject ();
			WebUtils.WriteJsonData (_context.Response, ref writer);
		}
	}
}