using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using Platform.Steam;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.UrlHandlers {
	public class SessionHandler : AbsHandler {
		
		private const string pageBasePath = "/app";
		private const string pageErrorPath = "/app/error/";

		private const string steamOpenIdVerifyUrl = "verifysteamopenid";
		private const string steamLoginUrl = "loginsteam";
		private const string steamLoginName = "Steam OpenID";
		private const string steamLoginFailedPage = "SteamLoginFailed";

		private const string userPassLoginUrl = "login";
		public const string userPassLoginName = "User/pass";
		private const string userPassErrorPage = "UserPassLoginFailed";

		public SessionHandler () : base (null) {
		}
		public override void HandleRequest (RequestContext _context) {
			if (_context.Request.RemoteEndPoint == null) {
				WebUtils.WriteText (_context.Response, "NoRemoteEndpoint", HttpStatusCode.BadRequest);
				return;
			}

			string subpath = _context.RequestPath.Remove (0, urlBasePath.Length);

			string remoteEndpointString = _context.Request.RemoteEndPoint!.ToString ();

			if (subpath.StartsWith (steamOpenIdVerifyUrl)) {
				if (HandleSteamVerification (parent.ConnectionHandler, _context, remoteEndpointString)) {
					_context.Response.Redirect (pageBasePath);
				} else {
					_context.Response.Redirect (pageErrorPath + steamLoginFailedPage);
				}
				return;
			}

			if (subpath.StartsWith ("logout")) {
				HandleLogout (parent.ConnectionHandler, _context, pageBasePath);
				return;
			}

			if (subpath.StartsWith (steamLoginUrl)) {
				HandleSteamLogin (_context, $"{urlBasePath}{steamOpenIdVerifyUrl}");
				return;
			}

			if (subpath.StartsWith (userPassLoginUrl)) {
				HandleUserPassLogin (parent.ConnectionHandler, _context, remoteEndpointString);
				return;
			}

			WebUtils.WriteText (_context.Response, "InvalidSessionsCommand", HttpStatusCode.BadRequest);
		}

		public static bool HandleUserPassLogin (ConnectionHandler _connectionHandler, RequestContext _context, string _remoteEndpointString) {
			if (!_context.Request.HasEntityBody) {
				WebUtils.WriteText (_context.Response, "NoLoginData", HttpStatusCode.BadRequest);
				return false;
			}

			Stream requestInputStream = _context.Request.InputStream;

			byte[] jsonInputData = new byte[_context.Request.ContentLength64];
			requestInputStream.Read (jsonInputData, 0, (int)_context.Request.ContentLength64);

			IDictionary<string, object> inputJson;
			try {
				inputJson = JsonSerializer.Deserialize<IDictionary<string, object>> (jsonInputData);
			} catch (Exception e) {
				Log.Error ("Error deserializing JSON from user/password login:");
				Log.Exception (e);
				WebUtils.WriteText (_context.Response, "InvalidLoginJson", HttpStatusCode.BadRequest);
				return false;
			}

			if (!inputJson.TryGetValue ("username", out object fieldNode) || fieldNode is not string username) {
				WebUtils.WriteText (_context.Response, "InvalidLoginJson", HttpStatusCode.BadRequest);
				return false;
			}

			if (!inputJson.TryGetValue ("password", out fieldNode) || fieldNode is not string password) {
				WebUtils.WriteText (_context.Response, "InvalidLoginJson", HttpStatusCode.BadRequest);
				return false;
			}

			if (!AdminWebUsers.Instance.TryGetUser (username, password, out AdminWebUsers.WebUser webUser)) {
				WebUtils.WriteText (_context.Response, "UserPassInvalid", HttpStatusCode.Unauthorized);
				Log.Out ($"[Web] User/pass login failed from {_remoteEndpointString}");
				return false;
			}

			bool loginResult = HandleUserIdLogin (_connectionHandler, _context, _remoteEndpointString, userPassLoginName, webUser.Name, webUser.PlatformUser, webUser.CrossPlatformUser);
			if (loginResult) {
				WebUtils.WriteText (_context.Response, "");
			} else {
				WebUtils.WriteText (_context.Response, "LoginError", HttpStatusCode.InternalServerError);
			}

			return loginResult;
		}

		public static void HandleSteamLogin (RequestContext _context, string _verificationCallbackUrl) {
			string host = $"{(WebUtils.IsSslRedirected (_context.Request) ? "https://" : "http://")}{_context.Request.UserHostName}";
			string url = OpenID.GetOpenIdLoginUrl (host, $"{host}{_verificationCallbackUrl}");
			_context.Response.Redirect (url);
		}

		public static bool HandleLogout (ConnectionHandler _connectionHandler, RequestContext _context, string _pageBase) {
			Cookie cookie = new Cookie ("sid", "", "/") {
				Expired = true
			};
			_context.Response.AppendCookie (cookie);

			if (_context.Connection == null) {
				_context.Response.Redirect (_pageBase);
				return false;
			}

			_connectionHandler.LogOut (_context.Connection.SessionID);
			_context.Response.Redirect (_pageBase);
			return true;
		}

		public static bool HandleSteamVerification (ConnectionHandler _connectionHandler, RequestContext _context, string _remoteEndpointString) {
			ulong id;
			try {
				id = OpenID.Validate (_context.Request);
			} catch (Exception e) {
				Log.Error ($"[Web] Error validating Steam login from {_remoteEndpointString}:");
				Log.Exception (e);
				return false;
			}

			if (id <= 0) {
				Log.Out ($"[Web] Steam OpenID login failed (invalid ID) from {_remoteEndpointString}");
				return false;
			}

			UserIdentifierSteam userId = new UserIdentifierSteam (id);
			return HandleUserIdLogin (_connectionHandler, _context, _remoteEndpointString, steamLoginName, userId.ToString (), userId);
		}

		public static bool HandleUserIdLogin (ConnectionHandler _connectionHandler, RequestContext _context, string _remoteEndpointString,
			string _loginName, string _username, PlatformUserIdentifierAbs _userId, PlatformUserIdentifierAbs _crossUserId = null) {
			try {
				WebConnection con = _connectionHandler.LogIn (_context.Request.RemoteEndPoint!.Address, _username, _userId, _crossUserId);

				int level1 = GameManager.Instance.adminTools.Users.GetUserPermissionLevel (_userId);
				int level2 = int.MaxValue;
				if (_crossUserId != null) {
					level2 = GameManager.Instance.adminTools.Users.GetUserPermissionLevel (_crossUserId);
				}

				int higherLevel = Math.Min (level1, level2);

				Log.Out ($"[Web] {_loginName} login from {_remoteEndpointString}, name {_username} with ID {_userId}, CID {(_crossUserId != null ? _crossUserId.ToString () : "none")}, permission level {higherLevel}");
				Cookie cookie = new Cookie ("sid", con.SessionID, "/") {
					Expired = false,
					Expires = DateTime.MinValue,
					HttpOnly = true,
					Secure = false
				};
				_context.Response.AppendCookie (cookie);

				return true;
			} catch (Exception e) {
				Log.Error ($"[Web] Error during {_loginName} login:");
				Log.Exception (e);
			}

			return false;
		}

	}
}