using Utf8Json;

namespace Webserver.WebAPI.APIs.ServerState {
	public abstract class KeyValueListAbs : AbsRestApi {
		private readonly UnityEngine.Profiling.CustomSampler buildSampler;

		private static readonly byte[] keyName = JsonWriter.GetEncodedPropertyNameWithBeginObject ("name");
		private static readonly byte[] keyType = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("type");
		private static readonly byte[] keyValue = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("value");
		private static readonly byte[] keyDefault = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("default");

		private int largestBuffer;

		protected KeyValueListAbs (string _listName) {
			buildSampler = UnityEngine.Profiling.CustomSampler.Create ($"JSON_{_listName}_BuildSampler");
		}

		protected override void HandleRestGet (RequestContext _context) {
			buildSampler.Begin ();

			PrepareEnvelopedResult (out JsonWriter writer);
			
			writer.EnsureCapacity (largestBuffer);
			writer.WriteBeginArray ();

			bool first = true;
			iterateList (ref writer, ref first);
			
			writer.WriteEndArray ();

			buildSampler.End ();

			int bufferContentSize = writer.CurrentOffset + 128;
			if (bufferContentSize > largestBuffer) {
				largestBuffer = bufferContentSize;
			}
			
			SendEnvelopedResult (_context, ref writer);
		}

		private void writeKeyType (ref JsonWriter _writer, ref bool _first, string _key, string _type) {
			if (!_first) {
				_writer.WriteValueSeparator ();
			}

			_first = false;
				
			_writer.WriteRaw (keyName);
			_writer.WriteString (_key);
				
			_writer.WriteRaw (keyType);
			_writer.WriteString (_type);
		}

		private void writeInt (ref JsonWriter _writer, ref bool _first, string _key, int _value) {
			writeKeyType (ref _writer, ref _first, _key, "int");
			_writer.WriteRaw (keyValue);
			_writer.WriteInt32 (_value);
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, int _value) {
			writeInt (ref _writer, ref _first, _key, _value);
			_writer.WriteEndObject ();
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, int _value, int? _default) {
			writeInt (ref _writer, ref _first, _key, _value);

			_writer.WriteRaw (keyDefault);
			if (_default.HasValue) {
				_writer.WriteInt32 (_default.Value);
			} else {
				_writer.WriteNull ();
			}

			_writer.WriteEndObject ();
		}

		
		private void writeFloat (ref JsonWriter _writer, ref bool _first, string _key, float _value) {
			writeKeyType (ref _writer, ref _first, _key, "float");
			_writer.WriteRaw (keyValue);
			_writer.WriteSingle (_value);
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, float _value) {
			writeFloat (ref _writer, ref _first, _key, _value);
			_writer.WriteEndObject ();
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, float _value, float? _default) {
			writeFloat (ref _writer, ref _first, _key, _value);

			_writer.WriteRaw (keyDefault);
			if (_default.HasValue) {
				_writer.WriteSingle (_default.Value);
			} else {
				_writer.WriteNull ();
			}

			_writer.WriteEndObject ();
		}

		
		private void writeBool (ref JsonWriter _writer, ref bool _first, string _key, bool _value) {
			writeKeyType (ref _writer, ref _first, _key, "bool");
			_writer.WriteRaw (keyValue);
			_writer.WriteBoolean (_value);
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, bool _value) {
			writeBool (ref _writer, ref _first, _key, _value);
			_writer.WriteEndObject ();
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, bool _value, bool? _default) {
			writeBool (ref _writer, ref _first, _key, _value);

			_writer.WriteRaw (keyDefault);
			if (_default.HasValue) {
				_writer.WriteBoolean (_default.Value);
			} else {
				_writer.WriteNull ();
			}

			_writer.WriteEndObject ();
		}

		
		private void writeString (ref JsonWriter _writer, ref bool _first, string _key, string _value) {
			writeKeyType (ref _writer, ref _first, _key, "string");
			_writer.WriteRaw (keyValue);
			_writer.WriteString (_value);
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, string _value) {
			writeString (ref _writer, ref _first, _key, _value);
			_writer.WriteEndObject ();
		}

		protected void addItem (ref JsonWriter _writer, ref bool _first, string _key, string _value, string _default) {
			writeString (ref _writer, ref _first, _key, _value);

			_writer.WriteRaw (keyDefault);
			_writer.WriteString (_default);

			_writer.WriteEndObject ();
		}

		protected abstract void iterateList (ref JsonWriter _writer, ref bool _first);

	}
}