#!/bin/bash

#   Copyright 2016 Christian 'Alloc' Illy
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.


. /usr/local/lib/7dtd/common.sh
. /usr/local/lib/7dtd/playerlog.sh

if [ $(isValidInstance $1) -eq 0 ]; then
	echo "No instance given or not a valid instance!"
	return
fi

INSTANCE=$1
LOGTIMESTAMP=$2
LOG=$(getInstancePath $INSTANCE)/logs/${LOGTIMESTAMP}_output_log.txt
CHATLOG=$(getInstancePath $INSTANCE)/logs/${LOGTIMESTAMP}_chat.log
COMMANDLOG=$(getInstancePath $INSTANCE)/logs/${LOGTIMESTAMP}_commandExecution.log

timestamp() {
	date '+%Y.%m.%d %H:%M:%S'
}

handleConnect() {
	local entityId="$1"
	local name="$2"
	local platformId="$3"
	local crossId="$4"
	local ip="$5"
	local ownerId="$6"
	
	logPlayerConnect $INSTANCE "$entityId" "$name" "$platformId" "$crossId" "$ip" "$ownerId"

	for H in $(getHooksFor playerConnect $INSTANCE); do
		$H $INSTANCE "$entityId" "$name" "$platformId" "$ip" "$ownerId" "$crossId"
	done
}

handleDisconnect() {
	local entityId="$1"

	logPlayerDisconnect $INSTANCE "$entityId"

	for H in $(getHooksFor playerDisconnect $INSTANCE); do
		$H $INSTANCE "$entityId" "$NICKNAME" "$PLATFORMID" "$IP" "$OWNERID" "$CROSSID"
	done
}

handlePlayerSpawnedInWorld() {
	local entityId="$1"
	local platformId="$2"
	local crossId="$3"
	local ownerId="$4"
	local playerName="$5"
	local reason="$6"
	local position="$7"
	
	for H in $(getHooksFor playerSpawned $INSTANCE); do
		$H $INSTANCE "$entityId" "$platformId" "$ownerId" "$playerName" "$reason" "$position" "$crossId"
	done
}

handleChat() {
	echo "$(timestamp): $1: $2 (SteamID $3, EntityID $4, Target $5)" >> $CHATLOG
	
	for H in $(getHooksFor chat $INSTANCE); do
		$H $INSTANCE "$1" "$2" "$3" "$4" "$5"
	done
}

handleGmsg() {
	echo "$(timestamp): GMSG: $1" >> $CHATLOG
	
	for H in $(getHooksFor gmsg $INSTANCE); do
		$H $INSTANCE "$1"
	done
}

handleRemoteCommand() {
	local cmd="$1"
	local name="$2"
	
	echo "$(timestamp): Player \"$name\" executed \"$cmd\"" >> $COMMANDLOG

	for H in $(getHooksFor remoteCommand $INSTANCE); do
		$H $INSTANCE "$cmd" "$name"
	done
}

handleTelnetCommand() {
	local cmd="$1"
	local ip="$2"

	echo "$(timestamp): Telnet from \"$ip\" executed \"$cmd\"" >> $COMMANDLOG

	for H in $(getHooksFor telnetCommand $INSTANCE); do
		$H $INSTANCE "$cmd" "$ip"
	done
}


if [ ! -d "$(getInstancePath $INSTANCE)/logs" ]; then
	mkdir "$(getInstancePath $INSTANCE)/logs"
fi

setAllPlayersOffline

rm $(getInstancePath $INSTANCE)/logs/current_output_log.txt
rm $(getInstancePath $INSTANCE)/logs/current_chat.log
rm $(getInstancePath $INSTANCE)/logs/current_commandExecution.log
ln -s $LOG $(getInstancePath $INSTANCE)/logs/current_output_log.txt
ln -s $CHATLOG $(getInstancePath $INSTANCE)/logs/current_chat.log
ln -s $COMMANDLOG $(getInstancePath $INSTANCE)/logs/current_commandExecution.log

sleep 5

NOBUF="stdbuf -e0 -o0"

$NOBUF tail -n 5000 -F $LOG |
$NOBUF tr '\\' '/' |
$NOBUF tr -d '\r' |
$NOBUF grep -v "^(Filename: " |
$NOBUF sed -r 's/^[0-9]+-[0-9]+-[0-9]+T[0-9]+:[0-9]+:[0-9]+ [0-9]+[.,][0-9]+ [A-Z]+ (.*)$/\1/' |
while read line ; do
	if [ -n "$line" ]; then
		#Player connected, entityid=1278, name=termo2, pltfmid=Steam_76561197997439820, crossid=EOS_eab...421, steamOwner=Steam_76561197997439820, ip=178.203.27.140 
		#Player connected, entityid=[0-9]*, name=.*, steamid=[0-9]*, steamOwner=[0-9]*, ip=[a-fA-F:0-9.]*$ 
		if [ -n "$(echo "$line" | grep '^Player connected,')" ]; then
			entityId=$(expr "$line" : 'Player connected, entityid=\([0-9]*\), name=.*, pltfmid=.*, crossid=.*, steamOwner=.*, ip=[a-fA-F:0-9.]*$')
			playerName=$(expr "$line" : 'Player connected, entityid=[0-9]*, name=\(.*\), pltfmid=.*, crossid=.*, steamOwner=.*, ip=[a-fA-F:0-9.]*$')
			platformId=$(expr "$line" : 'Player connected, entityid=[0-9]*, name=.*, pltfmid=\(.*\), crossid=.*, steamOwner=.*, ip=[a-fA-F:0-9.]*$')
			crossId=$(expr "$line" : 'Player connected, entityid=[0-9]*, name=.*, pltfmid=.*, crossid=\(.*\), steamOwner=.*, ip=[a-fA-F:0-9.]*$')
			steamOwner=$(expr "$line" : 'Player connected, entityid=[0-9]*, name=.*, pltfmid=.*, crossid=.*, steamOwner=\(.*\), ip=[a-fA-F:0-9.]*$')
			ip=$(expr "$line" : 'Player connected, entityid=[0-9]*, name=.*, pltfmid=.*, crossid=.*, steamOwner=.*, ip=\([a-fA-F:0-9.]*\)$')
			sleep 1
			handleConnect "$entityId" "$playerName" "$platformId" "$crossId" "$ip" "$steamOwner"
			unset entityId playerName platformId crossId steamOwner ip
		#Player disconnected: EntityID=[0-9]*, PlayerID='[0-9]*', OwnerID='[0-9]*', PlayerName='.*'$ 
		elif [ -n "$(echo "$line" | grep '^Player disconnected: ')" ]; then 
			entityId=$(expr "$line" : "Player disconnected: EntityID=\(.*\), PltfmId='.*$") 
			handleDisconnect "$entityId"
			unset entityId
		#PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=.*, PltfmId='.*', CrossId='.*', OwnerID='.*', PlayerName='.*', ClientNumber='.*'
		elif [ -n "$(echo "$line" | grep '^PlayerSpawnedInWorld ')" ]; then
			reason=$(expr "$line" : "PlayerSpawnedInWorld (reason: \(.*\), position: .*, .*, .*): EntityID=.*, PltfmId='.*', CrossId='.*', OwnerID='.*', PlayerName='.*', ClientNumber='.*'$") 
			position=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: \(.*, .*, .*\)): EntityID=.*, PltfmId='.*', CrossId='.*', OwnerID='.*', PlayerName='.*', ClientNumber='.*'$") 
			entityId=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=\(.*\), PltfmId='.*', CrossId='.*', OwnerID='.*', PlayerName='.*', ClientNumber='.*'$") 
			platformId=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=.*, PltfmId='\(.*\)', CrossId='.*', OwnerID='.*', PlayerName='.*', ClientNumber='.*'$") 
			crossId=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=.*, PltfmId='.*', CrossId='\(.*\)', OwnerID='.*', PlayerName='.*', ClientNumber='.*'$") 
			ownerId=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=.*, PltfmId='.*', CrossId='.*', OwnerID='\(.*\)', PlayerName='.*', ClientNumber='.*'$") 
			playerName=$(expr "$line" : "PlayerSpawnedInWorld (reason: .*, position: .*, .*, .*): EntityID=.*, PltfmId='.*', CrossId='.*', OwnerID='.*', PlayerName='\(.*\)', ClientNumber='.*'$") 
			handlePlayerSpawnedInWorld "$entityId" "$platformId" "$crossId" "$ownerId" "$playerName" "$reason" "$position"
			unset reason position entityId platformId crossId ownerId playerName
		#GMSG: .*$
		elif [ -n "$(echo "$line" | grep -E '^GMSG: .+')" ]; then
			msg=$(expr "$line" : 'GMSG: \(.*\)$')
			handleGmsg "$msg"
			unset msg
		#Chat (from '<steamid>', entity id '<entityid>', to '<targettype>'): '<senderchatname>': <msg>
		elif [ -n "$(echo "$line" | grep -E '^Chat .+')" ]; then
			steamId=$(expr "$line" : "Chat (from '\(.*\)', entity id '.*', to '.*'): '.*': .*$") 
			entityId=$(expr "$line" : "Chat (from '.*', entity id '\(.*\)', to '.*'): '.*': .*$") 
			targetType=$(expr "$line" : "Chat (from '.*', entity id '.*', to '\(.*\)'): '.*': .*$") 
			name=$(expr "$line" : "Chat (from '.*', entity id '.*', to '.*'): '\(.*\)': .*$") 
			msg=$(expr "$line" : "Chat (from '.*', entity id '.*', to '.*'): '.*': \(.*\)$") 
			handleChat "$name" "$msg" "$steamId" "$entityId" "$targetType"
			unset name msg steamId entityId targetType
		#Executing command ".*" from client ".*"$ 
		elif [ -n "$(echo "$line" | grep '^Executing command '.*' from client')" ]; then 
			cmd=$(expr "$line" : "Executing command '\(.*\)' from client .*$") 
			nick=$(expr "$line" : "Executing command '.*' from client \(.*\)$") 
			handleRemoteCommand "$cmd" "$nick"
			unset cmd nick
		#Executing command ".*" by Telnet from .*$ 
		elif [ -n "$(echo "$line" | grep '^Executing command '.*' by Telnet from ')" ]; then 
			cmd=$(expr "$line" : "Executing command '\(.*\)' by Telnet from .*$") 
			ip=$(expr "$line" : "Executing command '.*' by Telnet from \(.*\)$") 
			handleTelnetCommand "$cmd" "$ip"
			unset cmd ip
		fi
	fi
done
