using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class WebUsers : AbsRestApi {
		private const string propertyName = "name";
		private const string propertyPassword = "password";
		private const string propertyPlatformUserId = "platformUserId";
		private const string propertyCrossplatformUserId = "crossplatformUserId";

		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyName);
		private static readonly byte[] jsonKeyPlatformUserId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPlatformUserId);
		private static readonly byte[] jsonKeyCrossplatformUserId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyCrossplatformUserId);

		private static AdminWebUsers WebUsersInstance => AdminWebUsers.Instance;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);

			if (string.IsNullOrEmpty (id)) {
				writer.WriteBeginArray ();

				bool first = true;
				foreach ((_, AdminWebUsers.WebUser user) in WebUsersInstance.GetUsers ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeUserJson (ref writer, user);
				}

				writer.WriteEndArray ();

				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeUserJson (ref JsonWriter _writer, AdminWebUsers.WebUser _user) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_user.Name ?? "");
			_writer.WriteRaw (jsonKeyPlatformUserId);
			JsonCommons.WritePlatformUserIdentifier (ref _writer, _user.PlatformUser);
			_writer.WriteRaw (jsonKeyCrossplatformUserId);
			JsonCommons.WritePlatformUserIdentifier (ref _writer, _user.CrossPlatformUser);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!TryParseName (_context, _jsonInputData, out string userName)) {
				return;
			}

			if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPassword, out string password)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_PASSWORD);
				return;
			}

			if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPlatformUserId, out IDictionary<string, object> userIdField)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_PLATFORM_USER_ID);
				return;
			}

			if (!JsonCommons.TryReadPlatformUserIdentifier (userIdField, out PlatformUserIdentifierAbs platformUserId)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.INVALID_PLATFORM_USER_ID);
				return;
			}

			PlatformUserIdentifierAbs crossplatformUserId = null;
			
			if (JsonCommons.TryGetJsonField (_jsonInput, propertyCrossplatformUserId, out userIdField)) {
				if (!JsonCommons.TryReadPlatformUserIdentifier (userIdField, out crossplatformUserId)) {
					SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.INVALID_CROSSPLATFORM_USER_ID);
					return;
				}
			}

			WebUsersInstance.AddUser (userName, password, platformUserId, crossplatformUserId);

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			if (!TryParseName (_context, null, out string userName)) {
				return;
			}

			bool removed = WebUsersInstance.RemoveUser (userName);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		private bool TryParseName (RequestContext _context, byte[] _jsonInputData, out string _userName) {
			string id = _context.RequestPath;
			_userName = default;

			if (string.IsNullOrEmpty (id)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_USERNAME);
				return false;
			}

			_userName = id;
			return true;
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}