using System;
using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class Blacklist : AbsRestApi {
		private const string propertyName = "name";
		private const string propertyUserId = "userId";
		private const string propertyBannedUntil = "bannedUntil";
		private const string propertyBanReason = "banReason";

		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyName);
		private static readonly byte[] jsonKeyUserId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyUserId);
		private static readonly byte[] jsonKeyBannedUntil = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyBannedUntil);
		private static readonly byte[] jsonKeyBanReason = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyBanReason);

		private static AdminBlacklist BlacklistInstance => GameManager.Instance.adminTools.Blacklist;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);

			if (string.IsNullOrEmpty (id)) {
				writer.WriteBeginArray ();

				bool first = true;
				foreach (AdminBlacklist.BannedUser ban in BlacklistInstance.GetBanned ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeBan (ref writer, ban);
				}

				writer.WriteEndArray ();

				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeBan (ref JsonWriter _writer, AdminBlacklist.BannedUser _ban) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_ban.Name ?? "");
			_writer.WriteRaw (jsonKeyUserId);
			JsonCommons.WritePlatformUserIdentifier (ref _writer, _ban.UserIdentifier);
			_writer.WriteRaw (jsonKeyBannedUntil);
			JsonCommons.WriteDateTime (ref _writer, _ban.BannedUntil);
			_writer.WriteRaw (jsonKeyBanReason);
			_writer.WriteString (_ban.BanReason);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!TryParseId (_context, _jsonInputData, out PlatformUserIdentifierAbs userId)) {
				return;
			}

			if (!JsonCommons.TryReadDateTime (_jsonInput, propertyBannedUntil, out DateTime bannedUntil)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_BANNED_UNTIL);
				return;
			}

			JsonCommons.TryGetJsonField (_jsonInput, propertyBanReason, out string banReason);

			JsonCommons.TryGetJsonField (_jsonInput, propertyName, out string name);

			BlacklistInstance.AddBan (name, userId, bannedUntil, banReason);

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			if (!TryParseId (_context, null, out PlatformUserIdentifierAbs userId)) {
				return;
			}

			bool removed = BlacklistInstance.RemoveBan (userId);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		private bool TryParseId (RequestContext _context, byte[] _jsonInputData, out PlatformUserIdentifierAbs _userId) {
			string id = _context.RequestPath;
			_userId = default;

			if (string.IsNullOrEmpty (id)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_USER);
				return false;
			}

			if (!PlatformUserIdentifierAbs.TryFromCombinedString (id, out _userId)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.INVALID_USER);
				return false;
			}

			return true;
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}