using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class WebApiTokens : AbsRestApi {
		private const string propertyName = "name";
		private const string propertySecret = "secret";
		private const string propertyPermissionLevel = "permissionLevel";

		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyName);
		private static readonly byte[] jsonKeySecret = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertySecret);
		private static readonly byte[] jsonKeyPermissionLevel = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevel);

		private static AdminApiTokens ApiTokensInstance => AdminApiTokens.Instance;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);
			
			if (string.IsNullOrEmpty (id)) {
				
				writer.WriteBeginArray ();
			
				bool first = true;
				foreach ((_, AdminApiTokens.ApiToken token) in ApiTokensInstance.GetTokens ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeTokenJson (ref writer, token);
				}

				writer.WriteEndArray ();
				
				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeTokenJson (ref JsonWriter _writer, AdminApiTokens.ApiToken _token) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_token.Name);
			_writer.WriteRaw (jsonKeySecret);
			_writer.WriteString (_token.Secret);
			_writer.WriteRaw (jsonKeyPermissionLevel);
			_writer.WriteInt32 (_token.PermissionLevel);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			string id = _context.RequestPath;

			if (string.IsNullOrEmpty (id)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_NAME);
				return;
			}

			if (!JsonCommons.TryGetJsonField (_jsonInput, propertySecret, out string secret)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_SECRET);
				return;
			}

			if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPermissionLevel, out int permissionLevel)) {
				SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_PERMISSION_LEVEL);
				return;
			}

			ApiTokensInstance.AddToken (id, secret, permissionLevel);

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			string id = _context.RequestPath;

			bool removed = ApiTokensInstance.RemoveToken (id);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}