using System.Collections.Generic;
using System.Net;
using JetBrains.Annotations;
using Utf8Json;
using Webserver.Permissions;

namespace Webserver.WebAPI.APIs.Permissions {
	[UsedImplicitly]
	public class UserPermissions : AbsRestApi {
		private const string propertyName = "name";
		private const string propertyUserId = "userId";
		private const string propertyPermissionLevel = "permissionLevel";
		private const string propertyGroupId = "groupId";
		private const string propertyPermissionLevelMods = "permissionLevelMods";
		private const string propertyPermissionLevelNormal = "permissionLevelNormal";

		private static readonly byte[] jsonKeyUsers = JsonWriter.GetEncodedPropertyNameWithBeginObject ("users");
		private static readonly byte[] jsonKeyGroups = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("groups");

		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithBeginObject (propertyName);
		private static readonly byte[] jsonKeyUserId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyUserId);

		private static readonly byte[] jsonKeyPermissionLevel =
			JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevel);

		private static readonly byte[] jsonKeyGroupId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyGroupId);

		private static readonly byte[] jsonKeyPermissionLevelMods =
			JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevelMods);

		private static readonly byte[] jsonKeyPermissionLevelNormal =
			JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator (propertyPermissionLevelNormal);

		private static AdminUsers UsersInstance => GameManager.Instance.adminTools.Users;

		protected override void HandleRestGet (RequestContext _context) {
			string id = _context.RequestPath;

			PrepareEnvelopedResult (out JsonWriter writer);

			if (string.IsNullOrEmpty (id)) {
				writer.WriteRaw (jsonKeyUsers);
				writer.WriteBeginArray ();

				bool first = true;
				foreach ((_, AdminUsers.UserPermission userPermission) in UsersInstance.GetUsers ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeUserJson (ref writer, userPermission);
				}

				writer.WriteEndArray ();

				writer.WriteRaw (jsonKeyGroups);
				writer.WriteBeginArray ();

				first = true;
				foreach ((_, AdminUsers.GroupPermission groupPermission) in UsersInstance.GetGroups ()) {
					if (!first) {
						writer.WriteValueSeparator ();
					}

					first = false;

					writeGroupJson (ref writer, groupPermission);
				}

				writer.WriteEndArray ();

				writer.WriteEndObject ();

				SendEnvelopedResult (_context, ref writer);
				return;
			}

			writer.WriteRaw (WebUtils.JsonEmptyData);
			SendEnvelopedResult (_context, ref writer, HttpStatusCode.BadRequest);
		}

		private void writeUserJson (ref JsonWriter _writer, AdminUsers.UserPermission _userPermission) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_userPermission.Name ?? "");
			_writer.WriteRaw (jsonKeyUserId);
			JsonCommons.WritePlatformUserIdentifier (ref _writer, _userPermission.UserIdentifier);
			_writer.WriteRaw (jsonKeyPermissionLevel);
			_writer.WriteInt32 (_userPermission.PermissionLevel);
			_writer.WriteEndObject ();
		}

		private void writeGroupJson (ref JsonWriter _writer, AdminUsers.GroupPermission _groupPermission) {
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_groupPermission.Name ?? "");
			_writer.WriteRaw (jsonKeyGroupId);
			_writer.WriteString (_groupPermission.SteamIdGroup);
			_writer.WriteRaw (jsonKeyPermissionLevelMods);
			_writer.WriteInt32 (_groupPermission.PermissionLevelMods);
			_writer.WriteRaw (jsonKeyPermissionLevelNormal);
			_writer.WriteInt32 (_groupPermission.PermissionLevelNormal);
			_writer.WriteEndObject ();
		}

		protected override void HandleRestPost (RequestContext _context, IDictionary<string, object> _jsonInput, byte[] _jsonInputData) {
			if (!PermissionsApiHelpers.TryParseId (_context, _jsonInputData, out PlatformUserIdentifierAbs userId, out string groupId)) {
				return;
			}

			if (userId != null) {
				if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPermissionLevel, out int permissionLevel)) {
					SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_PERMISSION_LEVEL);
					return;
				}

				JsonCommons.TryGetJsonField (_jsonInput, propertyName, out string name);

				UsersInstance.AddUser (name, userId, permissionLevel);
			} else {
				if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPermissionLevelMods, out int permissionLevelMods)) {
					SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_PERMISSION_LEVEL_MODS);
					return;
				}

				if (!JsonCommons.TryGetJsonField (_jsonInput, propertyPermissionLevelNormal, out int permissionLevelNormal)) {
					SendEmptyResponse (_context, HttpStatusCode.BadRequest, _jsonInputData, EApiErrorCode.NO_OR_INVALID_PERMISSION_LEVEL_NORMAL);
					return;
				}

				JsonCommons.TryGetJsonField (_jsonInput, propertyName, out string name);

				UsersInstance.AddGroup (name, groupId, permissionLevelNormal, permissionLevelMods);
			}

			SendEmptyResponse (_context, HttpStatusCode.Created);
		}

		protected override void HandleRestDelete (RequestContext _context) {
			if (!PermissionsApiHelpers.TryParseId (_context, null, out PlatformUserIdentifierAbs userId, out string groupId)) {
				return;
			}

			bool removed = userId != null ? UsersInstance.RemoveUser (userId) : UsersInstance.RemoveGroup (groupId);

			SendEmptyResponse (_context, removed ? HttpStatusCode.NoContent : HttpStatusCode.NotFound);
		}

		protected override bool AllowPostWithId => true;

		public override int[] DefaultMethodPermissionLevels () => new[] {
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelInheritGlobal,
			AdminWebModules.MethodLevelNotSupported,
			AdminWebModules.MethodLevelInheritGlobal
		};
	}
}