using System.Collections.Generic;
using System.Net;
using AllocsFixes.PersistentData;
using JetBrains.Annotations;
using Utf8Json;
using Webserver;
using Webserver.WebAPI;

namespace AllocsFixes.WebAPIs {
	[UsedImplicitly]
	public class GetPlayerInventory : AbsWebAPI {
		public override void HandleRequest (RequestContext _context) {
			if (_context.Request.QueryString ["userid"] == null) {
				WebUtils.WriteText (_context.Response, "No user id given", HttpStatusCode.BadRequest);
				return;
			}

			string userIdString = _context.Request.QueryString ["userid"];
			if (!PlatformUserIdentifierAbs.TryFromCombinedString (userIdString, out PlatformUserIdentifierAbs userId)) {
				WebUtils.WriteText (_context.Response, "Invalid user id given", HttpStatusCode.BadRequest);
				return;
			}

			Player p = PersistentContainer.Instance.Players.GetByUserId (userId);
			if (p == null) {
				WebUtils.WriteText (_context.Response, "Unknown user id given", HttpStatusCode.NotFound);
				return;
			}

			GetInventoryArguments (_context, out bool showIconColor, out bool showIconName);

			JsonWriter writer = new JsonWriter ();
			DoPlayer (ref writer, p, showIconColor, showIconName);
			WebUtils.WriteJsonData (_context.Response, ref writer);
		}

		internal static void GetInventoryArguments (RequestContext _context, out bool _showIconColor, out bool _showIconName) {
			if (_context.Request.QueryString ["showiconcolor"] == null || !bool.TryParse (_context.Request.QueryString ["showiconcolor"], out _showIconColor)) {
				_showIconColor = true;
			}
			
			if (_context.Request.QueryString ["showiconname"] == null || !bool.TryParse (_context.Request.QueryString ["showiconname"], out _showIconName)) {
				_showIconName = true;
			}
		}

		private static readonly byte[] jsonKeyUserId = JsonWriter.GetEncodedPropertyNameWithBeginObject ("userid");
		private static readonly byte[] jsonKeyCrossPlatformId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("crossplatformid");
		private static readonly byte[] jsonKeyEntityId = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("entityid");
		private static readonly byte[] jsonKeyPlayerName = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("playername");
		private static readonly byte[] jsonKeyBag = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("bag");
		private static readonly byte[] jsonKeyBelt = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("belt");
		private static readonly byte[] jsonKeyEquipment = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("equipment");

		internal static void DoPlayer (ref JsonWriter _writer, Player _player, bool _showIconColor, bool _showIconName) {
			PersistentData.Inventory inv = _player.Inventory;
			
			_writer.WriteRaw (jsonKeyUserId);
			_writer.WriteString (_player.PlatformId.CombinedString);
			
			_writer.WriteRaw (jsonKeyCrossPlatformId);
			_writer.WriteString (_player.CrossPlatformId?.CombinedString ?? "");
			
			_writer.WriteRaw (jsonKeyEntityId);
			_writer.WriteInt32 (_player.EntityID);
			
			_writer.WriteRaw (jsonKeyPlayerName);
			_writer.WriteString (_player.Name);
			
			_writer.WriteRaw (jsonKeyBag);
			DoInventory (ref _writer, inv.bag, _showIconColor, _showIconName);
			
			_writer.WriteRaw (jsonKeyBelt);
			DoInventory (ref _writer, inv.belt, _showIconColor, _showIconName);
			
			_writer.WriteRaw (jsonKeyEquipment);
			DoEquipment (ref _writer, inv.equipment, _showIconColor, _showIconName);
			
			_writer.WriteEndObject ();
		}

		private static void DoInventory (ref JsonWriter _writer, IReadOnlyList<InvItem> _inv, bool _showIconColor, bool _showIconName) {
			_writer.WriteBeginArray ();
			for (int i = 0; i < _inv.Count; i++) {
				if (i > 0) {
					_writer.WriteValueSeparator ();
				}
				GetJsonForItem (ref _writer, _inv [i], _showIconColor, _showIconName);
			}
			_writer.WriteEndArray ();
		}

		private static void DoEquipment (ref JsonWriter _writer, IReadOnlyList<InvItem> _equ, bool _showIconColor, bool _showIconName) {
			_writer.WriteBeginObject ();
			
			AddEquipment (ref _writer, "head", _equ, EquipmentSlots.Head, _showIconColor, _showIconName);
			_writer.WriteValueSeparator ();

			AddEquipment (ref _writer, "armor", _equ, EquipmentSlots.Chest, _showIconColor, _showIconName);
			_writer.WriteValueSeparator ();

			AddEquipment (ref _writer, "boots", _equ, EquipmentSlots.Feet, _showIconColor, _showIconName);
			_writer.WriteValueSeparator ();

			AddEquipment (ref _writer, "gloves", _equ, EquipmentSlots.Hands, _showIconColor, _showIconName);
			
			_writer.WriteEndObject ();
		}

		private static void AddEquipment (ref JsonWriter _writer, string _slotname, IReadOnlyList<InvItem> _items, EquipmentSlots _slot, bool _showIconColor, bool _showIconName) {
			_writer.WritePropertyName (_slotname);

			int slotindex = (int)_slot;

			if (_items == null || _items[slotindex] == null) {
				// Slot not found / empty
				_writer.WriteNull ();
				return;
			}
			
			InvItem item = _items [slotindex];
			GetJsonForItem (ref _writer, item, _showIconColor, _showIconName);
		}

		private static readonly byte[] jsonKeyCount = JsonWriter.GetEncodedPropertyNameWithBeginObject ("count");
		private static readonly byte[] jsonKeyName = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("name");
		private static readonly byte[] jsonKeyIcon = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("icon");
		private static readonly byte[] jsonKeyIconColor = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("iconcolor");
		private static readonly byte[] jsonKeyQuality = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("quality");
		private static readonly byte[] jsonKeyQualityColor = JsonWriter.GetEncodedPropertyNameWithPrefixValueSeparator ("qualitycolor");

		private static void GetJsonForItem (ref JsonWriter _writer, InvItem _item, bool _showIconColor, bool _showIconName) {
			if (_item == null) {
				_writer.WriteNull ();
				return;
			}
			
			_writer.WriteRaw (jsonKeyCount);
			_writer.WriteInt32 (_item.count);
			
			_writer.WriteRaw (jsonKeyName);
			_writer.WriteString (_item.itemName);
			
			if (_showIconName) {
				_writer.WriteRaw (jsonKeyIcon);
				_writer.WriteString (_item.icon);
			}

			if (_showIconColor) {
				_writer.WriteRaw (jsonKeyIconColor);
				_writer.WriteString (_item.iconcolor);
			}

			_writer.WriteRaw (jsonKeyQuality);
			_writer.WriteInt32 (_item.quality);

			if (_item.quality >= 0) {
				_writer.WriteRaw (jsonKeyQualityColor);
				_writer.WriteString (QualityInfo.GetQualityColorHex (_item.quality));
			}
			
			_writer.WriteEndObject ();
		}
	}
}