using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Platform.EOS;

namespace AllocsFixes.PersistentData {
	[Serializable]
	public class Players {
		public readonly Dictionary<PlatformUserIdentifierAbs, Player> Dict = new Dictionary<PlatformUserIdentifierAbs, Player> ();

		public int Count => Dict.Count;

		public Player GetOrCreate (PlatformUserIdentifierAbs _internalId, PlatformUserIdentifierAbs _platformId, PlatformUserIdentifierAbs _crossPlatformId) {
			if (_internalId == null) {
				return null;
			}

			if (_internalId is UserIdentifierEos eosId) {
				string _ = eosId.ProductUserIdString;
			}

			if (Dict.TryGetValue (_internalId, out Player pOld)) {
				return pOld;
			}

			Log.Out ($"Created new player entry for ID: {_internalId}");
			Player p = new Player (_internalId, _platformId, _crossPlatformId);
			Dict.Add (_internalId, p);
			return p;
		}

		public Player GetByInternalId (PlatformUserIdentifierAbs _internalId) {
			if (_internalId == null) {
				return null;
			}

			return Dict.TryGetValue (_internalId, out Player pOld) ? pOld : null;
		}

		public Player GetByUserId (PlatformUserIdentifierAbs _userId) {
			foreach ((_, Player p) in Dict) {
				if (p.PlatformId.Equals (_userId) || p.CrossPlatformId.Equals (_userId)) {
					return p;
				}
			}

			return null;
		}

		public Player GetByString (string _nameOrId, bool _ignoreColorCodes) {
			if (string.IsNullOrEmpty (_nameOrId)) {
				return null;
			}

			if (PlatformUserIdentifierAbs.TryFromCombinedString (_nameOrId, out PlatformUserIdentifierAbs userId)) {
				return GetByUserId (userId);
			}

			if (int.TryParse (_nameOrId, out int entityId)) {
				foreach ((_, Player p) in Dict) {
					if (p.IsOnline && p.EntityID == entityId) {
						return p;
					}
				}
			}

			foreach ((_, Player p) in Dict) {
				string name = p.Name;
				if (_ignoreColorCodes) {
					name = Regex.Replace (name, "\\[[0-9a-fA-F]{6}\\]", "");
				}

				if (p.IsOnline && name.EqualsCaseInsensitive (_nameOrId)) {
					return p;
				}
			}

			return null;
		}
	}
}