#!/bin/bash

#   Copyright 2016 Christian 'Alloc' Illy
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.



sdtdSubcommandInstancesList() {
	printf "%-*s | %-*s | %-*s | %-*s | %-*s | %-*s\n" 20 "Instance name" 8 "Running" 7 "Players" 9 "Game Port" 11 "Dashb. Port" 6 "Telnet"
	printf -v line "%*s-+-%*s-+-%*s-+-%*s-+-%*s-+-%*s\n" 20 " " 8 " " 7 " " 9 " " 11 " " 6 " "
	echo ${line// /-}
	for I in $(getInstanceList); do
		if [ $(isRunning $I) -eq 1 ]; then
			run="yes"
			cur=$(telnetCommand $I lp | grep -aE "^\s?Total of " | cut -d\  -f 3)
		else
			run="no"
			cur="-"
		fi

		if [ -z $cur ]; then
			cur="?"
		fi
		max=$(getConfigValue $I ServerMaxPlayerCount)
		port=$(getConfigValue $I ServerPort)
		cpState="disabled"
		if [ "$(getConfigValue $I WebDashboardEnabled)" = "true" ]; then
			cpState="$(getConfigValue $I WebDashboardPort)"
		fi
		telnet=$(getConfigValue $I TelnetPort)

		printf "%-*s | %*s |   %2s/%2d | %9d | %*s | %*s\n" 20 "$I" 8 "$run" $cur $max $port 11 "$cpState" 6 "$telnet"
	done
}

sdtdSubcommandInstancesCreate() {
	while : ; do
		readInstanceName
		[ $(isValidInstance "$INSTANCE") -eq 0 ] && break
		echo "Instance name already in use."
		INSTANCE=
	done
	echo
	
	local IPATH=$(getInstancePath "$INSTANCE")
	mkdir -p "$IPATH" 2>/dev/null

	if [ $(configTemplateExists) -eq 1 ]; then
		local USETEMPLATE
		while : ; do
			read -p "Use the config template? [Yn] " USETEMPLATE
			USETEMPLATE=${USETEMPLATE:-Y}
			case $USETEMPLATE in
				y|Y)
					cp $SDTD_BASE/templates/config.xml $IPATH/config.xml
					break
					;;
				n|N)
					break
					;;
			esac
		done
		echo
	fi
	
	if [ ! -f $IPATH/config.xml ]; then
		cp $SDTD_BASE/engine/serverconfig.xml $IPATH/config.xml
	fi

	if [ -z "$EDITOR" ]; then
		echo "No EDITOR specified in the global configuration file (7dtd.conf)!"
		echo "Please add/edit the variable 'EDITOR' and set it to the text editor you would"
		echo "like to use to edit the XML configs, preferably an editor that supports XML"
		echo "syntax highlighting for easier editing."
		echo "Run '7dtd.sh instances edit $INSTANCE' afterwards to edit the configuration."
		echo
	else
		$EDITOR $IPATH/config.xml
	fi

	if [ -f "$SDTD_BASE/templates/admins.xml" ]; then
		cp "$SDTD_BASE/templates/admins.xml" "$IPATH/"
	fi
	chown -R $SDTD_USER:$SDTD_GROUP $IPATH
	echo "Done"
}

sdtdSubcommandInstancesEdit() {
	if [ -z "$EDITOR" ]; then
		echo "No EDITOR specified in the global configuration file (7dtd.conf)!"
		echo "Please add/edit the variable 'EDITOR' and set it to the text editor you would"
		echo "like to use to edit the XML configs, preferably an editor that supports XML"
		echo "syntax highlighting for easier editing."
		return
	fi
        
	if [ $(isValidInstance "$1") -eq 0 ]; then
		echo "No instance given or not a valid instance!"
		return
	fi
		
	if [ $(isRunning "$1") -eq 0 ]; then
		local INSTANCE=$1
		local IPATH=$(getInstancePath "$INSTANCE")

		$EDITOR $IPATH/config.xml
	else
		echo "Instance $1 is currently running. Please stop it first."
	fi
}

sdtdSubcommandInstancesDelete() {
	if [ $(isValidInstance "$1") -eq 0 ]; then
		echo "No instance given or not a valid instance!"
		return
	fi

	if [ $(isRunning "$1") -eq 0 ]; then
		local SECCODE=$(dd if=/dev/urandom bs=1 count=100 2>/dev/null \
			| tr -cd '[:alnum:]' | head -c5)
		local SECCODEIN
		echo
		echo "WARNING: Do you really want to delete the following instance?"
		echo "    $1"
		echo "This will delete all of its configuration and save data."
		echo "If you REALLY want to continue enter the following security code:"
		echo "    $SECCODE"
		echo
		read -p "Security code: " -e SECCODEIN
		if [ "$SECCODE" = "$SECCODEIN" ]; then
			rm -R "$(getInstancePath "$1")"
			echo "Done"
		else
			echo "Security code did not match, aborting."
		fi
	else
		echo "Instance $1 is currently running. Please stop it first."
	fi
}

sdtdSubcommandInstancesPrintConfig() {
	if [ $(isValidInstance "$1") -eq 0 ]; then
		echo "No instance given or not a valid instance!"
		return
	fi
		
	local INSTANCE=$1
	local IPATH=$(getInstancePath "$INSTANCE")
	xmlstarlet sel -t -m "//property" -v "@name" -o " = " -v "@value" -nl $IPATH/config.xml | sort
}

sdtdSubcommandInstancesPrintXml() {
	if [ $(isValidInstance "$1") -eq 0 ]; then
		echo "No instance given or not a valid instance!"
		return
	fi
		
	local INSTANCE=$1
	local IPATH=$(getInstancePath "$INSTANCE")
	
	which pygmentize > /dev/null 2>&1
	if [ $? -eq 0 ]; then
		cat $IPATH/config.xml | pygmentize -l xml
	else
		cat $IPATH/config.xml
	fi
}

sdtdCommandInstances() {
	SUBCMD=$1
	shift
	case $SUBCMD in
		list)
			sdtdSubcommandInstancesList "$@"
			;;
		create)
			sdtdSubcommandInstancesCreate "$@"
			;;
		edit)
			sdtdSubcommandInstancesEdit "$@"
			;;
		delete)
			sdtdSubcommandInstancesDelete "$@"
			;;
		print_config)
			sdtdSubcommandInstancesPrintConfig "$@"
			;;
		print_xml)
			sdtdSubcommandInstancesPrintXml "$@"
			;;
		*)
			sdtdCommandInstancesHelp
			;;
	esac
}

sdtdCommandInstancesHelp() {
	line() {
		printf "  %-*s %s\n" 19 "$1" "$2"
	}

	echo "Usage: $(basename $0) instances <subcommand>"
	echo "Subcommands are:"
	line "list" "List all defined instances and their status."
	line "create" "Create a new instance"
	line "edit <instance>" "Edit an existing instance"
	line "delete <instance>" "Delete an existing instance"
	line "print_config <instance>" "Print the current config of the instance to the console"
	line "print_xml <instance>" "Print the config XML as is"
}

sdtdCommandInstancesDescription() {
	echo "List all defined instances"
}

sdtdCommandInstancesExpects() {
	case $1 in
		2)
			echo "list create edit delete print_config print_xml"
			;;
		3)
			case $2 in
				edit|delete|print_config|print_xml)
					echo "$(getInstanceList)"
					;;
			esac
			;;
	esac
}

