using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Net;
using UnityEngine;
using Object = UnityEngine.Object;

namespace Webserver.UrlHandlers {
	public class ItemIconHandler : AbsHandler {
		private readonly Dictionary<string, byte[]> icons = new Dictionary<string, byte[]> ();
		private readonly bool logMissingFiles;

		private bool loaded;

		static ItemIconHandler () {
			Instance = null;
		}

		public ItemIconHandler (bool _logMissingFiles, string _moduleName = null) : base (_moduleName) {
			logMissingFiles = _logMissingFiles;
			Instance = this;
		}

		public static ItemIconHandler Instance { get; private set; }

		public override void HandleRequest (RequestContext _context) {
			if (!loaded) {
				_context.Response.StatusCode = (int) HttpStatusCode.InternalServerError;
				Log.Out ("[Web] IconHandler: Icons not loaded");
				return;
			}

			if (!_context.RequestPath.EndsWith (".png", StringComparison.OrdinalIgnoreCase)) {
				_context.Response.StatusCode = (int) HttpStatusCode.BadRequest;
				return;
			}

			string requestFileName = _context.RequestPath.Remove (0, urlBasePath.Length);
			int indexOfExtSep = requestFileName.LastIndexOf ('.');
			if (indexOfExtSep < 0) {
				_context.Response.StatusCode = (int) HttpStatusCode.BadRequest;
				return;
			}

			requestFileName = requestFileName.Remove (indexOfExtSep);

			if (!icons.TryGetValue (requestFileName, out byte[] icon)) {
				_context.Response.StatusCode = (int)HttpStatusCode.NotFound;
				if (logMissingFiles) {
					Log.Out ($"[Web] IconHandler: FileNotFound: \"{_context.RequestPath}\" ");
				}
				return;
			}

			_context.Response.ContentType = MimeType.GetMimeType (".png");

			_context.Response.ContentLength64 = icon.Length;
			_context.Response.OutputStream.Write (icon, 0, icon.Length);
		}

		private class LoadingStats {
			public int Files;
			public int Tints;
			public readonly MicroStopwatch MswTotal = new MicroStopwatch (false);
			public readonly MicroStopwatch MswLoading = new MicroStopwatch (false);
			public readonly MicroStopwatch MswEncoding = new MicroStopwatch (false);
			public readonly MicroStopwatch MswTinting = new MicroStopwatch (false);
		}

		private const int LoadingMaxMsPerFrame = 100;
		private bool loading;
		public IEnumerator LoadIcons () {
			
			lock (icons) {
				if (loading || loaded) {
					yield break;
				}

				loading = true;

				MicroStopwatch mswPerFrame = new MicroStopwatch (true);

				LoadingStats stats = new LoadingStats ();
				stats?.MswTotal.Start ();

				// Get list of used tints for all items
				Dictionary<string, List<Color>> tintedIcons = new Dictionary<string, List<Color>> ();
				foreach (ItemClass ic in ItemClass.list) {
					if (ic == null) {
						continue;
					}

					Color tintColor = ic.GetIconTint ();
					if (tintColor == Color.white) {
						continue;
					}

					string name = ic.GetIconName ();
					if (!tintedIcons.TryGetValue (name, out List<Color> tintsList)) {
						tintsList = new List<Color> ();
						tintedIcons.Add (name, tintsList);
					}

					tintsList.Add (tintColor);
				}

				yield return loadIconsFromFolder (GameIO.GetGameDir ("Data/ItemIcons"), tintedIcons, stats, mswPerFrame);

				// Load icons from mods
				foreach (Mod mod in ModManager.GetLoadedMods ()) {
					string modIconsPath = $"{mod.Path}/ItemIcons";
					yield return loadIconsFromFolder (modIconsPath, tintedIcons, stats, mswPerFrame);
				}
				
				loaded = true;

				if (stats == null) {
					Log.Out ($"[Web] IconHandler: Loaded {icons.Count} icons");
				} else {
					stats.MswTotal.Stop ();
					Log.Out ($"[Web] IconHandler: Loaded {icons.Count} icons ({stats.Files} source images with {stats.Tints} tints applied)");
					Log.Out ($"[Web] IconHandler: Total time {stats.MswTotal.ElapsedMilliseconds} ms, loading files {stats.MswLoading.ElapsedMilliseconds} ms, tinting files {stats.MswTinting.ElapsedMilliseconds} ms, encoding files {stats.MswEncoding.ElapsedMilliseconds} ms");

					int totalSize = 0;
					foreach ((string _, byte[] iconData) in icons) {
						totalSize += iconData.Length;
					}
					
					Log.Out ($"[Web] IconHandler: Cached {totalSize / 1024} KiB");
				}
			}
		}

		private IEnumerator loadIconsFromFolder (string _path, Dictionary<string, List<Color>> _tintedIcons, LoadingStats _stats,
			MicroStopwatch _mswPerFrame) {
			if (!Directory.Exists (_path)) {
				yield break;
			}

			_mswPerFrame.ResetAndRestart ();

			foreach (string file in Directory.GetFiles (_path)) {
				try {
					if (!file.EndsWith (".png", StringComparison.OrdinalIgnoreCase)) {
						continue;
					}

					string name = Path.GetFileNameWithoutExtension (file);
					Texture2D tex = new Texture2D (1, 1, TextureFormat.ARGB32, false);
					
					_stats?.MswLoading.Start ();
					byte[] sourceBytes = File.ReadAllBytes (file);
					if (!tex.LoadImage (sourceBytes)) {
						_stats?.MswLoading.Stop ();
						continue;
					}
					_stats?.MswLoading.Stop ();

					AddIcon (name, sourceBytes, tex, _tintedIcons, _stats);

					Object.Destroy (tex);
				} catch (Exception e) {
					Log.Error ($"[Web] Failed loading icon from {_path}");
					Log.Exception (e);
				}

				if (_mswPerFrame.ElapsedMilliseconds >= LoadingMaxMsPerFrame) {
					yield return null;
					_mswPerFrame.ResetAndRestart ();
				}
			}
		}

		private void AddIcon (string _name, byte[] _sourceBytes, Texture2D _tex, Dictionary<string, List<Color>> _tintedIcons, LoadingStats _stats) {
			_stats?.MswEncoding.Start ();
			icons [$"{_name}__FFFFFF"] = _sourceBytes;
			_stats?.MswEncoding.Stop ();

			if (_stats != null) {
				_stats.Files++;
			}

			if (!_tintedIcons.TryGetValue (_name, out List<Color> tintsList)) {
				return;
			}

			foreach (Color c in tintsList) {
				string tintedName = $"{_name}__{c.ToHexCode ()}";
				if (icons.ContainsKey (tintedName)) {
					continue;
				}

				Texture2D tintedTex = new Texture2D (_tex.width, _tex.height, TextureFormat.ARGB32, false);

				_stats?.MswTinting.Start ();
				TextureUtils.ApplyTint (_tex, tintedTex, c);
				_stats?.MswTinting.Stop ();

				_stats?.MswEncoding.Start ();
				icons [tintedName] = tintedTex.EncodeToPNG ();
				_stats?.MswEncoding.Stop ();

				Object.Destroy (tintedTex);

				if (_stats != null) {
					_stats.Tints++;
				}
			}
		}
		
	}
}