#!/bin/bash

PLAYERSXML=$(getInstancePath $1)/players.xml
PLAYERSLOG=$(getInstancePath $1)/logs/$(date '+%Y-%m-%d_%H-%M-%S')_players.log

timestamp() {
	date '+%Y.%m.%d %H:%M:%S'
}

# Create empty player list if not existing
createPlayerList() {
	if [ ! -f "$PLAYERSXML" ]; then
		echo "<Players/>" > $PLAYERSXML
	fi
	if [ -z "$(cat $PLAYERSXML)" ]; then
		echo "<Players/>" > $PLAYERSXML
	fi
}

# Set all players for an instance to offline (on startup/shutdown)
setAllPlayersOffline() {
	createPlayerList
	$XMLSTARLET ed -L \
		-u "/Players/Player/@online" -v "false" \
		"$PLAYERSXML"
}

# Handle a player connect for logging/tracking
# Params:
#   1: Instance name
#   2: Entity ID
#   3: Steam ID
#   4: Nick name
logPlayerConnect() {
	ENTITYID="$2"
	STEAMID="$3"
	NICKNAME="$4"

	echo "$(timestamp) +++ $ENTITYID $NICKNAME $STEAMID" >> "$PLAYERSLOG"

	createPlayerList
	
	XPATHBASE="/Players/Player[@steamid='$STEAMID']"

	if [ -z $($XMLSTARLET sel -t -v "$XPATHBASE/@steamid" "$PLAYERSXML") ]; then
		$XMLSTARLET ed -L \
			-s "/Players" -t elem -n "Player" -v "" \
			-i "/Players/Player[not(@steamid)]" -t attr -n "steamid" -v "$STEAMID" \
			-i "$XPATHBASE" -t attr -n "nick" -v "$NICKNAME" \
			-i "$XPATHBASE" -t attr -n "playtime" -v "0" \
			-i "$XPATHBASE" -t attr -n "logins" -v "1" \
			-i "$XPATHBASE" -t attr -n "lastlogin" -v "$(date '+%s')" \
			-i "$XPATHBASE" -t attr -n "online" -v "true" \
			-i "$XPATHBASE" -t attr -n "entityid" -v "$ENTITYID" \
			"$PLAYERSXML"
	else
		LOGINS=$($XMLSTARLET sel -t -v "$XPATHBASE/@logins" "$PLAYERSXML")
		(( LOGINS++ ))
		$XMLSTARLET ed -L \
			-u "$XPATHBASE/@lastlogin" -v "$(date '+%s')" \
			-u "$XPATHBASE/@online" -v "true" \
			-u "$XPATHBASE/@entityid" -v "$ENTITYID" \
			-u "$XPATHBASE/@logins" -v "$LOGINS" \
			"$PLAYERSXML"
	fi
}

# Handle a player disconnect for logging/tracking
# Params:
#   1: Instance name
#   2: Entity ID
logPlayerDisconnect() {
	ENTITYID="$2"

	createPlayerList

	XPATHBASE="/Players/Player[@entityid='$ENTITYID'][@online='true']"

	if [ -f $PLAYERSXML ]; then
		if [ ! -z $($XMLSTARLET sel -t -v "$XPATHBASE/@steamid" "$PLAYERSXML") ]; then
			NICKNAME=$($XMLSTARLET sel -t -v "$XPATHBASE/@nick" "$PLAYERSXML")
			STEAMID=$($XMLSTARLET sel -t -v "$XPATHBASE/@steamid" "$PLAYERSXML")
			LOGINTIME=$($XMLSTARLET sel -t -v "$XPATHBASE/@lastlogin" "$PLAYERSXML")
			PLAYTIME=$($XMLSTARLET sel -t -v "$XPATHBASE/@playtime" "$PLAYERSXML")
			NOW=$(date '+%s')
			PLAYTIME=$(( PLAYTIME + NOW - LOGINTIME ))
			$XMLSTARLET ed -L \
				-u "$XPATHBASE/@playtime" -v "$PLAYTIME" \
				-u "$XPATHBASE/@online" -v "false" \
				"$PLAYERSXML"
		fi
	fi

	echo "$(timestamp) --- $ENTITYID $NICKNAME $STEAMID" >> "$PLAYERSLOG"
}

